/*
	
	Reason for making a canvas based virtual keyboard instead of a html-button's based:
	* Slighly faster to render
	* Can be rendered on-top of the main canvas, where html elements on-top of a canvas will slow down rendering.
	* Easier to ajust buttons so they fill the whole screen, compared to CSS and old browser support
	
	Problem: Smartphone screens are way too small for touch typing. 4x9 buttons work great, but it gets problematic if you need 5x20
	Solution: Use a stulys !? Not many mobile phones support a stylus though
	
	Problem: The virtual keyboard is too large to fit on the screen.
	Solution: Hide the file tabs while the virtual keyboard is visible !?
	
	
	Q: Why not use the device's keyboard !?
	A: Programming on it is a pain as keys souch as {[]}" is hard to reach
	
	Note: We will need two views: one for when the device is in horizontal, and one for when it's vertical!
	
	Problem: When a widget is fired, there's simply no more space available, can't have both the keyboard and the widget visible at the same time.
	
	
	Problem: When clicking on an input element inside a widget, the device wants to bring up the devices's virtual keyboard
	
	Long-press: Inserts the character, then auto-complete
	
	We tried having alt-keys, but typing was too slow, where alt keys required 3 key presses.
	
	Most common characters, beside the normal latin letters:
	space=6772002 
	Enter=986014 
	,=679396 
	.=591280 
	)=506646 
	(=506640 
	==456621 
	"=378701 
	;=374216 
	/=284895 
	0=275398 
	'=265540 
	-=224118 
	1=206276 
	{=203136 
	}=202922 
	_=198407 
	:=195979 
	=165540 
	2=154382 
	*=129766 
	+=123478 
	3=119448 
	[=99911 
	]=99547 
	4=99258 
	5=97924 
	6=89885 
	7=81554 
	8=79697 
	9=71607 
	\=69728 
	&=68828 
	>=57232 
	<=56023 
	!=55563 
	|=51660 
	`=33476 
	?=29721 
	@=24256 
	$=17894 
	%=12639 
	#=10033 
	^=4949  
	
	Don't forget about the clipboard!
	
	Split keyboard!?
	
	Symbols: http://xahlee.info/comp/unicode_computing_symbols.html
	
*/

(function() {

	"use strict";
	
	var canvas = document.createElement("canvas");
	var ctx = canvas.getContext('2d', {alpha: false, antialias: false});
	var pixelRatio = 1;
	var buttonWithToHeightRatio = 1.5;
	var buttonWidth = 10;
	var buttonHeight = 10;
	var CAPS = false;
	var buttonLocations = [];
	var buttons = [];
	var canvasWidth = 500;
	var canvasHeight = 200;
	var maxButtonsPerRow = 10;
	var radius = 8;
	var margin = 4;
	var lineWidth = 1;
	var buttonsPerRow = [0,0,0,0,0,0];
	var verticalLayout = [];
	var horizontalLayout = [];
	var totalRows = 3;
	var menuItem;
	// Each key can have 3 alternative functions, depending if alt1, alt2 or both alt1 and alt2 is active
	var ALT1 = false;
	var ALT2 = false;
	var clickTimer;
	var customAltKeys = [];
	var builtinName = "Builtin";
	var nativeName = "Native";
	var oldCanvasHeight = 0; // Optimization: Don't resize and re-render if the size is the same as before
	var oldCanvasWidth = 0;
	var useNative = false; // is native keyboard activated
	var useBuiltin = false; // is built-in keyboard activated
	var usePhysical = false;
	var lastUsedKeyboard = "builtin"; // builtin, native or physical
	var nativeKeyboardCatcher;
	var winMenuVirtual, winMenuOnScreen, winMenuPhysical;
	var winMenuVibration, vibrationEnabled = true;
	var checkActiveElementInterval;
	
	canvas.onmousedown = canvasMouseDown;
	canvas.onmouseup = canvasMouseUp;
	canvas.ontouchstart = canvasMouseDown;
	canvas.ontouchend = canvasMouseUp;
	
	canvas.style.position = "relative";
	canvas.style.zIndex = 4; // Above the dashboard, but below the context menu, and below the window menu's
	
	var labelShowBuiltin = S("virtual_keyboard");
	var labelShowNative = S("native_keyboard");
	
	var winMenuKeyboard;
	
	EDITOR.plugin({
		desc: "Testing canvas based virtual keyboard",
		load: function loadVirtualKeyboard() {
			
			// Wait for touch events before showing the virtual keyboard
			EDITOR.addEvent( "mouseClick", {dir: "down", fun: vrkeyboardMouseDown, targetClass:"fileCanvas", order: 1000} );
			EDITOR.addEvent( "mouseClick", {dir: "up", fun: vrkeyboardMouseUp, targetClass:"fileCanvas", order: 1000} );
			EDITOR.addEvent( "mouseClick", {dir: "up", fun: vrkeyboardDetectElement, order: 1000} );
			
			EDITOR.on("beforeResize", virtualKeyboardClaimHeight);
			EDITOR.on("afterResize", resizeVirtualKeyboard);
			EDITOR.on("hideVirtualKeyboard", hideMyVirtualKeyboards);
			EDITOR.on("showVirtualKeyboard", showMyVirtualKeyboards);
			
			EDITOR.on("fileOpen", showVirtualKeyboardMaybe);
			
			addButtons();
			
			menuItem = EDITOR.ctxMenu.add(labelShowBuiltin, toggleBetweenKeyboards, 26);
			winMenuKeyboard = EDITOR.windowMenu.add(labelShowBuiltin, [S("View"), 120], toggleBetweenKeyboards);
			
			winMenuVirtual = EDITOR.windowMenu.add(S("virtual"), [S("Editor"), S("keyboard_input")], menuPickVirtual);
			winMenuOnScreen = EDITOR.windowMenu.add(S("native_onscreen"), [S("Editor"), S("keyboard_input")], menuPickOnScreen);
			winMenuPhysical = EDITOR.windowMenu.add(S("physical_keyboard"), [S("Editor"), S("keyboard_input")], menuPickPhysical);
			winMenuVibration = EDITOR.windowMenu.add(S("vibration"), [S("Editor"), S("keyboard_input")], toggleVibration);
			
			EDITOR.on("registerAltKey", updateAltKey);
			EDITOR.on("unregisterAltKey", removeAltKey);
			EDITOR.on("storageReady", loadVirtualKeyboardSettings);
			
			var wrapper = document.getElementById("virtualKeyboard2");
			wrapper.style.display="none";
			wrapper.appendChild(canvas);
			
			setTimeout(function() {
				var input = EDITOR.input
				//alertBox("input=" + input);
			}, 3000);
			
		},
		unload: function unloadVirtualKeyboard() {
			
			EDITOR.removeEvent("mouseClick", vrkeyboardMouseDown);
			EDITOR.removeEvent("mouseClick", vrkeyboardMouseUp);
			
			EDITOR.removeEvent("beforeResize", virtualKeyboardClaimHeight);
			EDITOR.removeEvent("afterResize", resizeVirtualKeyboard);
			EDITOR.removeEvent("hideVirtualKeyboard", hideMyVirtualKeyboards);
			EDITOR.removeEvent("showVirtualKeyboard", showMyVirtualKeyboards);
			
			EDITOR.removeEvent("fileOpen", showVirtualKeyboardMaybe);
			
			EDITOR.removeEvent("registerAltKey", updateAltKey);
			EDITOR.removeEvent("storageReady", loadVirtualKeyboardSettings);
			
			hideBuiltinKeyboard();
			hideNativeKeyboard();
			
			EDITOR.ctxMenu.remove(menuItem);
			
			EDITOR.windowMenu.remove(winMenuKeyboard);
			EDITOR.windowMenu.remove(winMenuVirtual);
			EDITOR.windowMenu.remove(winMenuOnScreen);
			EDITOR.windowMenu.remove(winMenuPhysical);
			EDITOR.windowMenu.remove(winMenuVibration);
			
			var wrapper = document.getElementById("virtualKeyboard2");
			wrapper.removeChild(canvas);
			
		}
	});
	
	function vrkeyboardDetectElement(mouseX, mouseY, caret, mouseDirection, button, target, keyboardCombo, mouseUpEvent) {
		
		/*
			This will pull up the native onscreen keyboard.
			We don't want both.
			It's easier to hide our keyboard then the native keyboard.
		*/
		
		hideVirtualKeyboardIfTextElementIsFocused(target);
		
		return ALLOW_DEFAULT;
	}
	
	function hideVirtualKeyboardIfTextElementIsFocused(el) {
		if(el == undefined) el = document.activeElement;
		
		if(isTextBox(el)) hideBuiltinKeyboard();
	}
	
	function isTextBox(element) {
		if(typeof element != "object") return false;
		if(!element.tagName) return false;
		
		var tagName = element.tagName.toLowerCase();
		if (tagName === 'textarea') return true;
		if (tagName !== 'input') return false;

		var type = element.getAttribute('type');
		var type = (typeof type == "string") && type.toLowerCase();
		// if any of these input types is not supported by a browser, it will behave as input type text.
		var inputTypes = ['text', 'password', 'number', 'email', 'tel', 'url', 'search', 'date', 'datetime', 'datetime-local', 'time', 'month', 'week']
		return inputTypes.indexOf(type) >= 0;
	}
	
	function loadVirtualKeyboardSettings() {
		
		var storageVibration = EDITOR.storage.getItem("virtual_keyboard_vibration");
		if(storageVibration=="1") vibrationEnabled = true;
		else if(storageVibration)  vibrationEnabled = false;
		
		if( vibrationEnabled ) winMenuVibration.activate();
		else winMenuVibration.deactivate();
	}
	
	function toggleVibration() {
		vibrationEnabled = !vibrationEnabled;
		
		if(EDITOR.storage.ready()) EDITOR.storage.setItem("virtual_keyboard_vibration", (vibrationEnabled?"1":"0"));
		
		if( vibrationEnabled ) winMenuVibration.activate();
		else winMenuVibration.deactivate();
		
		winMenuVibration.hide();
	}
	
	function menuPickVirtual() {
		if(useBuiltin) {
			// It's already active, so deactivate it
			hideBuiltinKeyboard();
			lastUsedKeyboard = "builtin";
			winMenuVirtual.deactivate();
			return;
		}
		
		useNative = false;
		//useBuiltin = true;
		usePhysical = false;
		lastUsedKeyboard = "builtin"; // builtin, native or physical
		
		winMenuVirtual.activate();
		winMenuOnScreen.deactivate();
		winMenuPhysical.deactivate();
		
		EDITOR.ctxMenu.update(menuItem, true, labelShowNative);
		EDITOR.windowMenu.update(winMenuKeyboard, {active: true, label: labelShowNative});
		
		hideNativeKeyboard();
		showBuiltinKeyboard(); // Will set useBuiltin to true
	}
	
	function menuPickOnScreen() {
		if(useNative) {
			// It's already active, so deactivate it
			hideNativeKeyboard();
			lastUsedKeyboard = "native";
			winMenuOnScreen.deactivate();
			return;
		}
		//useNative = true;
		useBuiltin = false;
		usePhysical = false;
		lastUsedKeyboard = "native"; // builtin, native or physical
		
		winMenuVirtual.deactivate();
		winMenuOnScreen.activate();
		winMenuPhysical.deactivate();
		
		EDITOR.ctxMenu.update(menuItem, true, labelShowBuiltin);
		EDITOR.windowMenu.update(winMenuKeyboard, {active: true, label: labelShowBuiltin});
		
		hideBuiltinKeyboard();
		showNativeKeyboard(); // Will set useNative to true
	}
	
	function menuPickPhysical() {
		useNative = false;
		useBuiltin = false;
		usePhysical = true;
		lastUsedKeyboard = "physical"; // builtin, native or physical
		
		winMenuVirtual.deactivate();
		winMenuOnScreen.deactivate();
		winMenuPhysical.activate();
		
		hideNativeKeyboard();
		hideBuiltinKeyboard();
		
		EDITOR.ctxMenu.update(menuItem, false, labelShowBuiltin);
		EDITOR.windowMenu.update(winMenuKeyboard, {active: false, label: labelShowBuiltin});
		
		winMenuPhysical.hide();
		
		// If the key presses are not captured; Android will do weird annoying stuff.
		EDITOR.input = true;
		setTimeout(function() {
			EDITOR.input = true;
		}, 2000);
		
	}
	
	function showVirtualKeyboardMaybe() {
		if(EDITOR.touchScreen && !usePhysical) {
			showBuiltinKeyboard();
		}
	}
	
	function toggleBetweenKeyboards() {
		console.log("toggleBetweenKeyboards: useNative=" + useNative + " useBuiltin=" + useBuiltin);
		
		EDITOR.ctxMenu.hide();
		
		if(useNative) {
			// Don't show any
			hideNativeKeyboard();
			usePhysical = true;
			EDITOR.ctxMenu.update(menuItem, false, labelShowBuiltin);
			EDITOR.windowMenu.update(winMenuKeyboard, {active: false, label: labelShowBuiltin});
		}
		else if(useBuiltin) {
			// Show native
			hideBuiltinKeyboard();
			showNativeKeyboard();
			usePhysical = false;
			winMenuPhysical.deactivate();
		}
		else if(!useNative && !useBuiltin) {
			// Show built-in
			showBuiltinKeyboard();
			usePhysical = false;
			winMenuPhysical.deactivate();
		}
		else {
			throw new Error("This should never happen! useNative=" + useNative + " useBuiltin=" + useBuiltin);
		}
	}
	
	function toggleToNative() {
		if(useBuiltin) hideBuiltinKeyboard();
		
		showNativeKeyboard();
	}
	
	function showBuiltinKeyboard() {
		if(useBuiltin) {
			console.log("Builtin already active!");
			return;
		}
		
		var wrapper = document.getElementById("virtualKeyboard2");
		wrapper.style.display="block";
		
		EDITOR.ctxMenu.update(menuItem, true, labelShowNative);
		EDITOR.windowMenu.update(winMenuKeyboard, {active: true, label: labelShowNative});
		
		useBuiltin = true;
		
		if(EDITOR.currentFile) {
			// Wait for the resize, then scroll to the caret
			setTimeout(function() {
				EDITOR.currentFile.scrollToCaret();
			}, 500);
		}
		
		EDITOR.resizeNeeded();
		
		checkActiveElementInterval = setInterval(hideVirtualKeyboardIfTextElementIsFocused, 1000);
		
		EDITOR.stat("virtual_keyboard");
	}
	
	function hideBuiltinKeyboard() {
		console.log("Hiding builtin virtual keyboard!");
		var wrapper = document.getElementById("virtualKeyboard2");
		wrapper.style.display="none";
		EDITOR.resizeNeeded();
		useBuiltin = false;
		
		clearInterval(checkActiveElementInterval);
	}
	
	function showNativeKeyboard() {
		if(!useNative) {
EDITOR.ctxMenu.update(menuItem, true, labelShowNative);
			EDITOR.windowMenu.update(winMenuKeyboard, {active: true, label: labelShowNative});
		}
		// Always trigger native, even if already in use
		
		var keyboardCatcher = document.getElementById("keyboardCatcher");
		
		// Trigger native keyboard
		keyboardCatcher.focus();
		keyboardCatcher.click();
		
		// Input should go to current file
		EDITOR.input = true;
		
		useNative = true;
	}
	
	function hideNativeKeyboard() {
		var keyboardCatcher = document.getElementById("keyboardCatcher");
		keyboardCatcher.blur();
		
		useNative = false;
	}
	
	function hideMyVirtualKeyboards(keyboards) {
		var wasActive = [];
		
		if(useBuiltin) {
			wasActive.push(builtinName);
			hideBuiltinKeyboard();
			lastUsedKeyboard = "builtin";
		}
		
		if(useNative) {
			wasActive.push(nativeName);
			hideNativeKeyboard();
			lastUsedKeyboard = "native";
		}
		
		return wasActive;
	}
	
	function showMyVirtualKeyboards(keyboards) {
		if(!Array.isArray(keyboards)) throw new Error("keyboards need to be an array of keyboard names (" + nativeName + ", " + builtinName + ")");
		
		if(keyboards.indexOf(nativeName)) showNativeKeyboard();
		if(keyboards.indexOf(builtinName)) showBuiltinKeyboard();
		
		return keyboards;
	}
	
	function removeAltKey(fun) {
		var id = 0;
		var altNr = 0;
		for (var i=0; i<customAltKeys.length; i++) {
			if(customAltKeys[i].fun == fun) {
				
				id = customAltKeys[i].id;
				altNr = customAltKeys[i].alt;

				customAltKeys.splice(i, 1);
				
				delete verticalLayout[id]["alt" + altNr];
				
				console.log("Removed registeredAltKeys for " + UTIL.getFunctionName(fun) + " from " + verticalLayout[i].char + " !");
				
				return;
			}
		}
		
		console.warn("No match for " + UTIL.getFunctionName(fun) + " in registeredAltKeys!");
	}
	
	function updateAltKey(key) {
		console.log("updateAltKey: char=" + key.char + " alt=" + key.alt);
if(key.alt > 3) {
			console.log("Virtual keyboard only allow 3 alternatives per button!");
return false;
}
		for (var i=0, button; i<verticalLayout.length; i++) {
			button = verticalLayout[i];
			if(button.char == key.char) {
				console.log("Found char=" + key.char);
				if(verticalLayout[i]["alt" + key.alt] != undefined) {
					throw new Error("There is already something registered on " + button.char + " for alt=" + key.alt + " ! Unable to register function " + UTIL.getFunctionName(key.fun));
					return false;
				}
				else {
					verticalLayout[i]["alt" + key.alt] = key.label;
					customAltKeys.push({id: i, alt: key.alt, fun: key.fun});
					console.log("Added " + UTIL.getFunctionName(key.fun) + " as alt" + key.alt + " on " + key.char + " !");
					return true;
				}
			}
		}
		console.log("Could not find char=" + key.char + " on the virtual keyboard!");
		return false;
	}
	
	function virtualKeyboardClaimHeight(file, windowWidth, windowHeight) {
		
		/*
			Called before resize.
			Claim the height needed!
		*/
		
		if(!useBuiltin) return;
		
		buttons = verticalLayout;
		
		canvasWidth = windowWidth;
		
		if(windowWidth > windowHeight) {
			var orientation = "horizontal";
			buttonWithToHeightRatio = 0.9;
			//buttons = horizontalLayout;
			
			radius = canvasWidth / 200; // Rounded corners
			margin = canvasWidth / 500;
			lineWidth = canvasWidth / 2000;
		}
		else {
			var orientation = "vertical";
			buttonWithToHeightRatio = 2;
			
			radius = canvasWidth / 150; // Rounded corners
			margin = canvasWidth / 300;
			lineWidth = canvasWidth / 1500;
		}
		
		buttonsPerRow = calcButtonsPerRow(buttons);
		
		if(buttons.length == 0) throw new Error("No buttons found! windowWidth=" + windowWidth + " windowHeight=" + windowHeight + " (orientation=" + orientation + ") horizontalLayout.length=" + horizontalLayout.length + " verticalLayout.length=" + verticalLayout.length);
		if(buttonsPerRow[0] == 0) throw new Error("First row has no buttons! buttonsPerRow=" + JSON.stringify(buttonsPerRow) + " buttons=" + JSON.stringify(buttons, null, 2));
		if(buttonsPerRow[1] == 0) throw new Error("Second row has no buttons! buttonsPerRow=" + JSON.stringify(buttonsPerRow) + " buttons=" + JSON.stringify(buttons, null, 2));
		if(buttonsPerRow[2] == 0) throw new Error("Third row has no buttons! buttonsPerRow=" + JSON.stringify(buttonsPerRow) + " buttons=" + JSON.stringify(buttons, null, 2));
		
		console.log("maxButtonsPerRow=" + maxButtonsPerRow);
		console.log("buttonsPerRow=" + JSON.stringify(buttonsPerRow));
		console.log("totalRows=" + totalRows);
		
		
		buttonWidth = canvasWidth / maxButtonsPerRow;
		
		buttonHeight = buttonWithToHeightRatio * buttonWidth;
		canvasHeight = Math.ceil(buttonHeight * totalRows + margin);
		
		
		
		var wrapper = document.getElementById("virtualKeyboard2");
		wrapper.style.width=canvasWidth + "px";
		wrapper.style.height=canvasHeight + "px";
		
		
		//EDITOR.virtualKeyboardHeight = canvasHeight;
		
		console.log("virtualKeyboardClaimHeight: canvasWidth=" + canvasWidth + " buttonWidth=" + buttonWidth + " canvasHeight=" + canvasHeight + " buttonHeight=" + buttonHeight);
		
	}
	
	function resizeVirtualKeyboard(file, windowWidth, windowHeight) {
		// Called after a resize event
		
		if(!useBuiltin) return;
		
		if(canvasHeight == oldCanvasHeight && canvasWidth == oldCanvasWidth) {
			console.log("resizeVirtualKeyboard: No need to re-render! canvasHeight=" + canvasHeight + " oldCanvasHeight=" + oldCanvasHeight + " canvasWidth=" + canvasWidth + " oldCanvasWidth=" + oldCanvasWidth);
			return;
		}
		
		// debug
		var wrapper = document.getElementById("virtualKeyboard2");
		var wrapperBefore = wrapper.offsetWidth + "x" + wrapper.offsetHeight;
		
		var pixelRatio = window.devicePixelRatio || 1;
		
		canvas.width = canvasWidth * pixelRatio;
		canvas.height = canvasHeight * pixelRatio;
		// Setting the width and height will clear the canvas!
		ctx.restore();
		ctx.save();
		ctx.scale(pixelRatio,pixelRatio);
		//ctx.scale(1,1);
		ctx.textBaseline = "middle"; // Will also be reset when setting canvas.width!
		
		canvas.style.width=canvasWidth + "px";
		canvas.style.height=canvasHeight + "px";
		
		renderVirtualKeyboard();
		
		oldCanvasHeight = canvasHeight;
		oldCanvasWidth = canvasWidth;
		
		// debug
		var wrapperAfter = wrapper.offsetWidth + "x" + wrapper.offsetHeight;
		console.log("resizeVirtualKeyboard: pixelRatio=" + pixelRatio + " windowWidth=" + windowWidth + " windowHeight=" + windowHeight + " actual window size=" + window.innerWidth + "x" + window.innerHeight + " canvasWidth=" + canvasWidth + " canvasHeight=" + canvasHeight + " wrapperBefore=" + wrapperBefore + " wrapperAfter=" + wrapperAfter);
	}
	
	
	function calcButtonsPerRow(buttons) {
		
		if(buttons.length == 0) throw new Error("buttons.length=" + buttons.length);
		
		var buttonsPerRow = [0,0,0,0,0,0,0];
		
		for (var i=0; i<buttons.length; i++) {
			buttonsPerRow[buttons[i].row] += buttons[i].width;
		}
		
		for (var i=0; i<buttonsPerRow.length; i++) {
			if(buttonsPerRow[i] == 0) {
				totalRows = i;
				break;
			}
		}
		
		maxButtonsPerRow = Math.max(buttonsPerRow[0], buttonsPerRow[1], buttonsPerRow[2], buttonsPerRow[3], buttonsPerRow[4], buttonsPerRow[5]);
		
		if(maxButtonsPerRow == 0) throw new Error("maxButtonsPerRow=" + maxButtonsPerRow + " buttonsPerRow=" + JSON.stringify(buttonsPerRow));
		if(isNaN(maxButtonsPerRow)) throw new Error("maxButtonsPerRow=" + maxButtonsPerRow + " buttonsPerRow=" + JSON.stringify(buttonsPerRow));
		
		return buttonsPerRow;
		
	}
	
	function renderVirtualKeyboard() {
		/*
			
			Optimization: Measured when clicking ALt keys
			Naive/Original: 1.4 - 2.6 ms on Chrome
			Cache background: 1.9 - 3.5 on Chrome. Huh? Seems like copying from another canvas cost a lot!
			Use translate: 
		*/
		console.time("renderVirtualKeyboard");
		
		if(buttons.length == 0) throw new Error("buttons.length=" + buttons.length);
		
		/*
			if(pixelRatio !== 1) {
			ctx.restore();
			ctx.save();
			ctx.scale(pixelRatio,pixelRatio);
			//ctx.scale(1,1);
			}
		*/
		
		// Background
		ctx.fillStyle = "#000000";
		ctx.fillRect(0, 0, canvasWidth, canvasHeight);
		
		var cX = 0;
		var cY = 0;
		var x1 = 0;
		var y1 = 0;
		var x2 = 0;
		var y2 = 0;
		var gradient;
		var startX = 0;
		var accumulatedWidth = 0;
		var lastRow = 0;
		
		// ### Button backgrounds
		
		ctx.strokeStyle="white";
		ctx.lineWidth=lineWidth;
		
		for (var i=0; i<buttons.length; i++) {
			
			if(buttons[i].row != lastRow) {
				lastRow = buttons[i].row;
				accumulatedWidth = 0;
			}
			
			startX = (canvasWidth - (buttonsPerRow[buttons[i].row]) * buttonWidth) / 2;
			
			x1 = startX + accumulatedWidth + margin;
			y1 = (buttons[i].row+1) * buttonHeight - buttonHeight + margin;
			
			x2 = startX + accumulatedWidth + buttons[i].width*buttonWidth - margin*2;
			y2 = (buttons[i].row+1) * buttonHeight - margin*2;
			
			accumulatedWidth += buttons[i].width * buttonWidth;
			
			//cX = startX + buttons[i].col * buttonWidth - buttonWidth/2 - margin;
			//cY = (buttons[i].row+1) * buttonHeight - buttonHeight/2 - margin;
			
			// A path with rounded corners
			ctx.beginPath();
			ctx.moveTo(x1 + radius, y1);
			ctx.lineTo(x2 - radius, y1);
			ctx.quadraticCurveTo(x2, y1, x2, y1 + radius);
			ctx.lineTo(x2, y2 - radius);
			ctx.quadraticCurveTo(x2, y2, x2 - radius, y2);
			ctx.lineTo(x1 + radius, y2);
			ctx.quadraticCurveTo(x1, y2, x1, y2 - radius);
			ctx.lineTo(x1, y1 + radius);
			ctx.quadraticCurveTo(x1, y1, x1 + radius, y1);
			ctx.closePath();
			
			
			// A gradient background
			//gradient=ctx.createLinearGradient(cX-buttonWidth/2, cY-buttonHeight/2, buttonWidth, buttonHeight);
			gradient=ctx.createLinearGradient(x2, y1, x2, y2);
			gradient.addColorStop(0,"#656565");
			
			if(ALT1 && ALT2 && buttons[i].highlightAlt3) gradient.addColorStop(1,"orange");
			else if(ALT1 && buttons[i].highlightAlt1) gradient.addColorStop(1,"green");
			else if(ALT2 && buttons[i].highlightAlt2) gradient.addColorStop(1,"yellow");
			else gradient.addColorStop(1,"black");
			
			ctx.fillStyle=gradient;
			
			ctx.fill(); // Fill the path with rounded corners
			ctx.stroke();
			
			//ctx.fillRect(cX-buttonWidth/2+margin, cY-buttonHeight/2+margin, buttonWidth-margin*2, buttonHeight-margin*2);
			
			//ctx.drawImage(canvas, m, m, buttonWidth, buttonHeight, cX-buttonWidth/2+m, cY-buttonHeight/2+m, buttonWidth, buttonHeight);
		}
		
		// ### Button letters
		ctx.fillStyle = "#FFFFFF";
		ctx.textAlign = "center";
		ctx.font=  Math.floor(buttonHeight * 0.6)  + "px Arial";
		
		buttonLocations.length = 0;
		
		
		ctx.beginPath();
		ctx.lineWidth="1";
		ctx.strokeStyle="red";
		
		var text = "";
		var alt1 = "";
		var alt2 = "";
		var textWidth = 0;
		var textHeight = 0;
		var textArr = [];
		var textMeasure;
		var fontSize = 0;
		
		// Note: ctx.measureText() only has one property: width (not height)
		
		for (var i=0; i<buttons.length; i++) {
			
			startX = (canvasWidth - buttonsPerRow[buttons[i].row] * buttonWidth) / 2;
			
			fontSize = Math.floor(buttonHeight * 0.4 * buttons[i].textSize)
			
			ctx.font = fontSize + "px Arial";
			//textWidth = ctx.measureText(comment.count.toString()).width;
			
			if(buttons[i].row != lastRow) {
				lastRow = buttons[i].row;
				accumulatedWidth = 0;
			}
			
			cX = startX + accumulatedWidth + buttonWidth*buttons[i].width/2 - margin/2;
			cY = (buttons[i].row+1) * buttonHeight - buttonHeight/2 - margin/2;
			
			accumulatedWidth += buttons[i].width * buttonWidth;
			
			buttonLocations.push({id: i, x: cX, y: cY, width: buttons[i].width});
			
			//ctx.rect(cX-buttonWidth/2+margin, cY-buttonHeight/2+margin, buttonWidth-margin*2, buttonHeight-margin*2);
			
			
			
			if(ALT1 && ALT2 && buttons[i].alt3) {
				text = (CAPS && buttons[i].fun == normalButton) ? buttons[i].alt3.toUpperCase() : buttons[i].alt3;
			}
			else if(ALT1 && !ALT2 && buttons[i].alt1) {
				text = (CAPS && buttons[i].fun == normalButton) ? buttons[i].alt1.toUpperCase() : buttons[i].alt1;
			}
			else if(ALT2 && !ALT1 && buttons[i].alt2) {
				text = (CAPS && buttons[i].fun == normalButton) ? buttons[i].alt2.toUpperCase() : buttons[i].alt2;
			}
			else if(!ALT1 && !ALT2 && !(ALT1 && ALT2)) {
				text = (CAPS && buttons[i].fun == normalButton) ? buttons[i].char.toUpperCase() : buttons[i].char;
			}
			else text = null;
			
			if(text) printText(text, cX, cY, fontSize);
			
			alt1 = buttons[i].alt1 && (CAPS && buttons[i].fun == normalButton) ? buttons[i].alt1.toUpperCase() : buttons[i].alt1;
			alt2 = buttons[i].alt2 && (CAPS && buttons[i].fun == normalButton) ? buttons[i].alt2.toUpperCase() : buttons[i].alt2;
			
			//if(alt1 && ALT1) alt1 = (CAPS && buttons[i].fun == normalButton) ? buttons[i].char.toUpperCase() : buttons[i].char;
			//if(alt2 && ALT2) alt2 = (CAPS && buttons[i].fun == normalButton) ? buttons[i].char.toUpperCase() : buttons[i].char;
			
			if( (alt1 || alt2) && (!ALT1 && !ALT2) ) {
				
				fontSize = Math.floor(buttonHeight * 0.2 * buttons[i].textSize);
				
				ctx.font = fontSize + "px Arial";
				//textWidth = ctx.measureText(comment.count.toString()).width;
				
				if(alt1) printText(alt1, cX, cY-buttonHeight/3 + margin, fontSize, true); // ctx.fillText(alt1, cX, cY-buttonHeight/3 + margin);
				if(alt2) printText(alt2, cX, cY+buttonHeight/3 + margin, fontSize, true); // ctx.fillText(alt2, cX, cY+buttonHeight/3 - margin);
			}
		}
		ctx.stroke();
		buttonLocations.sort(sortLocationsX);
		
		
		console.timeEnd("renderVirtualKeyboard");
		
		function printText(text, cX, cY, fontSize, noSplit) {
			var textArr = [text];
			
			// Make sure the text fits
			var textWidth = ctx.measureText(text).width;
			var maxWidth = (buttonWidth*buttons[i].width+margin*2);
			if(textWidth > maxWidth || fontSize < 10) {
				console.warn("printText: text=" + text + " fontSize=" + fontSize + " textWidth=" + textWidth + " buttonWidth=" + buttonWidth + " noSplit=" + noSplit);
				
				if(!noSplit) {
					// Attempt to split it
					textArr = text.split(" ");
					textWidth = ctx.measureText(textArr[0]).width;
				}
				
				if(textWidth > maxWidth) {
					// Still too wide, use smaller font
					fontSize = Math.floor(fontSize * 0.8 * maxWidth / textWidth);
					ctx.font =  fontSize + "px Arial";
					console.log("Using fontSize=" + fontSize + " for text=" + text + " because textWidth was " + textWidth);
				}
			}
			
			for (var j=0; j<textArr.length; j++) {
				ctx.fillText( textArr[j], cX, cY + j*(fontSize+margin) - (textArr.length-1) * fontSize );
			}
		}
		
	}
	
	function sortLocationsX(a, b) {
		if(a.x > b.x) return -1;
		else if(b.x > a.x) return 1;
		else return 0; 
	}
	
	function canvasMouseDown(mouseDownEvent) {
		
		mouseDownEvent.preventDefault();
		mouseDownEvent.stopPropagation();
		
		if (vibrationEnabled && navigator.vibrate) {
			// vibration API supported
			var vibrateSuccess = navigator.vibrate([1]);
		}
		
		if(vibrationEnabled && !vibrateSuccess) EDITOR.beep(0.1, 120, "sine", 39);
		
		return false;
	}
	
	//var COL = 0;
	function canvasMouseUp(mouseUpEvent) {
		
		mouseUpEvent.preventDefault();
		mouseUpEvent.stopPropagation();
		
		//if(COL > 20) COL = 0;
		//return EDITOR.renderColumn(0, COL++, "G");
		
		//return clickButton(buttonLocations[0].id, mouseUpEvent);
		
		var click = getMouseLocation(mouseUpEvent);
		
		// Locations are sorted in X axis
		for (var i=0; i<buttonLocations.length; i++) {
			
			//console.log("click.x=" + click.x + " click.y=" + click.y + " button.x=" + buttonLocations[i].x + " button.y=" + buttonLocations[i].y + " buttonWidth=" + buttonWidth + " buttonHeight=" + buttonHeight + " ");
			
			if( click.x > (buttonLocations[i].x - buttonWidth * buttonLocations[i].width / 2)  &&  click.x < (buttonLocations[i].x + buttonWidth * buttonLocations[i].width / 2) && 
			click.y > (buttonLocations[i].y - buttonHeight/2)  &&  click.y < (buttonLocations[i].y + buttonHeight/2) ) return clickButton(buttonLocations[i].id, mouseUpEvent);
		}
		
		return true;
	}
	
	var COL = 0;
	function clickButton(id, someEvent) {
		var button = buttons[id];
		var customFunction;
		
		//return button.fun();
		
		var altNr = 0;
		
		if(ALT1 && ALT2) altNr = 3;
		else if(ALT1) altNr = 1;
		else if(ALT2) altNr = 2;
		
		for (var i=0; i<customAltKeys.length; i++) {
			if(customAltKeys[i].id == id && customAltKeys[i].alt == altNr) {
				customFunction = customAltKeys[i].fun;
				console.log("Using custom function " + UTIL.getFunctionName(customFunction) + " from alt keys!");
				break;
			}
		}
		
		if(customFunction) {
			// Send same parameters as keyBindings! (file, combo, character, charCode, direction, targetElementClass, keyDownEvent)
			var file = EDITOR.currentFile;
			var combo = {shift: false, alt: false, ctrl: false, sum: 0};
			var character = button.char;
			var charCode = CAPS ? button.charCodeCaps : button.charCode;
			var direction = "down";
			var targetElementClass = someEvent.target.className;
			customFunction(file, combo, character, charCode, direction, targetElementClass, someEvent); // Give same parameters as keybindings
		}
		else button.fun(); // Need to be button.fun so that the function will get the proper this
		
		
		clearTimeout(clickTimer);
		
		if(customFunction) {
			restore();
		}
		else if(!button.alt) {
			// To save one click you automatically go back to normal
			// But give enough time to allow many clicks
			clickTimer = setTimeout(restore, 500);
		}
		
		return false;
		
		function restore() {
			if(ALT1 || ALT2) {
				ALT1 = false;
				ALT2 = false;
				renderVirtualKeyboard();
			}
		}
		
	}
	
	// Buttons that have a function specified need to handle ALT1, ALT2, ALT2 && ALT2, and CAPS in that function
	// If ALT is a special function, the key need to have a function specified.
	function normalButton() {
		var button  = this;
		if(ALT1 && ALT2 && button.alt3) fireKey( CAPS ? button.alt3.toUpperCase().charCodeAt(0) : button.alt3.toLowerCase().charCodeAt(0) );
		else if(ALT1 && button.alt1) fireKey( CAPS ? button.alt1.toUpperCase().charCodeAt(0) : button.alt1.toLowerCase().charCodeAt(0) );
		else if(ALT2 && button.alt2) fireKey(CAPS ? button.alt2.toUpperCase().charCodeAt(0) : button.alt2.toLowerCase().charCodeAt(0) );
		else fireKey(CAPS ? button.charCodeCaps : button.charCode);
	}
	
	function fireKey(charCode, eventType) {
		
		/*
			Note: KeyCode is NOT the same as charCode!!
			
		*/
		
		console.log("fireKey: charCode=" + charCode + " eventType=" + eventType +
		" document.activeElement: id=" + document.activeElement.id + " node=" + document.activeElement.nodeName +
		" EDITOR.lastElementWithFocus: id=" + EDITOR.lastElementWithFocus.id + " node=" + EDITOR.lastElementWithFocus.nodeName);
		
		if(charCode == 8592) var KeyCode = 37; // ← left
		if(charCode == 8594) var KeyCode = 39; // → right
		if(charCode == 8593) var KeyCode = 38; // ↑ up
		if(charCode == 8595) var KeyCode = 40; // ↓ down
		
		if(KeyCode) eventType = "keydown";
		
		//event.preventDefault();
		
		if(eventType == undefined) eventType = "keypress";
		
		var el = document.activeElement;
		
		//if(document.activeElement != EDITOR.lastElementWithFocus) el = EDITOR.lastElementWithFocus;
		
		console.log("el: id=" + el.id + " node=" + el.nodeName + " type=" + el.type);
		
		if(EDITOR.isTextInputElement(el)) {
EDITOR.typeIntoElement(el, charCode);
		}
		else {
			EDITOR.input = true;
			}
		// I tried to optimize response time (it's very slow on Nokia N9) but while the first character displays fast, the next is slow due to the cpu being busy ...
		//EDITOR.renderColumn(EDITOR.currentFile.caret.row, EDITOR.currentFile.caret.col, key, EDITOR.settings.style.textColor);
		//setTimeout(defaultAction, 0); // Needed in order to make the canvas update!
		//function defaultAction() {
		var doDefaultAction = EDITOR.mock( eventType, { charCode: KeyCode || charCode } );
		console.log("eventType=" + eventType + " doDefaultAction=" + doDefaultAction);
		//}
		
	}
	
	
	
	
	function addButtons() {
		
		
		buttonsPerRow = [0,0,0,0,0,0];
		
		/*
			
			
		// ## Horizontal
		var row = 0;
		var col = 0;
		var orientation = "horizontal";
		
		add("1");
		add("2");
		add("3");
		add("4");
		add("5");
		add("6");
		add("7");
		add("8");
		add("9");
		add("0");
		
		add("(");
		add(")");
		
		add("[");
		add("]");
		
		add("back", {
			fun: function space(click) {
			fireKey(8, "keydown");
			return false;
		},
			charCode: 8,
			width: 2,
			textSize: 0.8
		});
		
		
		add("-");
		add("+");
		
		add("#");
		
		
		// ### Horizontal second row
		row = 1;
		col = 0
		
		add("@");
		add("q");
		add("w");
		add("e");
		add("r");
		add("t");
		add("y");
		add("u");
		add("i");
		add("o");
		add("p");
		add("å");
		
		add("=");
		
		add("'");
		add('"');
		
		add("/");
		add("*");
		add("%");
		add("~");
		
		
		// ### Horizontal third row
		row = 2;
		col = 0
		
		add("CAPS", {
fun: function capsLock(click) {
				CAPS = !CAPS;
				renderVirtualKeyboard();
			},
			charCode: -1,
			width: 1.5,
			textSize: 0.8
});
		
		add("a");
		add("s");
		add("d");
		add("f");
		add("g");
		add("h");
		add("j");
		add("k");
		add("l");
		add("ö");
		add("ä");
		
		add(".");
		
		add(";");
		
		add("{");
		
		add("Enter", {
fun: function space(click) {
				fireKey(13, "keydown");
			},
			charCode: 13,
			width: 1.5,
			textSize: 0.8
		});
		
		add("}");
		
		add("\\");
		
		
		// ### Horizontal fourth row
		row = 3;
		col = 0
		
		add("^");
		
		add("&");
		
		
		add("z");
		add("x");
		add("c");
		add("v");
		add("b");
		add("n");
		add("m");
		
		add("space", {
			charCode: 32,
			width: 1.5,
			textSize: 0.8
});
		
		add("compl", {
fun: function space(click) {
				fireKey(EDITOR.settings.autoCompleteKey, "keydown");
			},
			charCode: -1,
			width: 1.5,
			textSize: 0.8
});
		
		add("<");
		add(">");
		
		add(",");
		add("!");
		add("?");
		add(":");
		
		
		add("|");
		
		*/
		
		
		
		
		
		
		
		
		
		
		
		// ## Vertical
		var row = 0;
		var col = 0
		var orientation = "vertical";
		
		add("1", {           alt2: "~", alt3: "¹", width: 1.1}); // 206276
		
		add("q", {alt1: "2", alt2: "@", alt3: "²"});
		add("w", {alt1: "3", alt2: "#", alt3: "³"});
		add("e", {alt1: "4", alt2: "$", alt3: "£"});
		add("r", {alt1: "5", alt2: "%", atl3: "®"});
		add("t", {           alt2: "^", alt3: "þ"});
		
		add('"', {alt1: "'", alt2: "`", alt3: "་", width: 1.6}); // 378701
		add("&", {width: 1.2}); // 195979
		
		add("y", {                      alt3: "ü"});
		add("u", {alt1: "6",            alt3: "µ"});
		add("i", {alt1: "7",            alt3: "í"});
		add("o", {alt1: "8",            alt3: "¤"});
		add("p", {alt1: "9", alt2: "ö"});
		
		add("0", {width: 1.2}); // 275398
		
		
		// ### Vertical second row
		row = 1;
		col = 0
		
		add("(", {alt1: "{", alt2: "[", width: 1.5}); // 506640
		
		add("a", {alt1: "ä",            alt3: "á"});
		add("s", {alt1: "å",            alt3: "§"});
		add("d", {                      alt3: "€"});
		add("f", {                      alt3: "т"});
		add("g", {                      alt3: "я"});
		
		add("=", {width: 1.4}); // 456621
		
		add("ABC", { // Could use 🔠 but not supported by all browsers
			fun: function capsLock(click) {
				CAPS = !CAPS;
				renderVirtualKeyboard();
			},
			charCode: -1,
			width: 1.6,
			textSize: 0.8
		});
		
		add("h", {alt1: "←",    alt3: "н"}); // move left
		add("j", {alt1: "↓"}); // move down
		add("k", {alt1: "↑",    alt3: "к"}); // move up
		add("l", {alt1: "→"}); // move right
		
		add(")", {alt1: "}", alt2: "]", width: 1.6}); // 506646
		
		
		// ### Vertical third row
		row = 2;
		col = 0;
		
		add("copy", {
			fun: function copyPaste(click) {
				if(!EDITOR.currentFile) return;
				
				if(ALT1) {
					var clipboardData = EDITOR.getClipboardContent(gotClipboard);
				}
				else if(ALT2) {
					EDITOR.putIntoClipboard(EDITOR.currentFile.getSelectedText());
					EDITOR.currentFile.deleteSelection();
				}
				else {
					EDITOR.putIntoClipboard(EDITOR.currentFile.getSelectedText());
				}
				
				function gotClipboard(err, data) {
					if(err) alertBox(err.message);
					else EDITOR.mock("paste", {data: data});
				}
			},
			charCode: 8,
			width: 1.2,
			textSize: 0.8,
			alt1: "paste",
			alt2: "cut"
		});
		
		add("!", {width: 0.8}); // 55563
		
		add("z", {                      alt3: "œ"});
		add("x", {                      alt3: "¢"});
		add("c", {                      alt3: "©"});
		add("v");
		add("b", {                      alt3: "в"});
		
		add("/", {alt1: "\\", alt2: "|", width: 1}); // 284895
		add("+", {width: 1}); // 123478
		add("-", {alt1: "_", width: 1}); // 198407
		
		
		add("n", {                       alt3: "π"}); // n might be used as bigint annotator
		add("m", {                       alt3: "м"});
		
		add("*", {alt1: "?"}); // 129766
		
		
		
		add("back", {
			fun: function back(click) {
				fireKey(8, "keydown");
				return false;
			},
			charCode: 8,
			width: 2.1,
			textSize: 0.8,
		});
		
		
		
		// ### Vertical fourth row
		row = 3;
		col = 0;
		
		add("Compl", {
			fun: function autocomplete(click) {
				if(ALT1) toggleToNative();
				else if(ALT2) hideBuiltinKeyboard();
				else fireKey(EDITOR.settings.autoCompleteKey, "keydown");
			},
			charCode: EDITOR.settings.autoCompleteKey,
			width: 1.3,
			textSize: 0.7,
			alt1: "Native",
			alt2: "Done"
		});
		
		add(";", {width: 1.4}); // 374216
		
		add("Alt-1", {
			fun: function alternate1() {
				ALT1=!ALT1;
				renderVirtualKeyboard();
			},
			charCode: -1,
			highlightWhenActive: true,
			width: 1.5,
			textSize: 0.6,
			highlightAlt1: true,
			highlightAlt3: true,
			alt: true
		});
		
		add(",", {width: 1.5, textSize: 1.5}); // 679396
		
		add("space", {
			fun: function space(click) {
				fireKey(32, "keypress"); // space
			},
			charCode: 32,
			width: 2.4,
			textSize: 0.7,
		});
		
		add(".", {alt1: ":", width: 1.5, textSize: 1.5}); // 591280
		
		add("Alt-2", {
			fun: function alternate1() {
				ALT2=!ALT2;
				renderVirtualKeyboard();
			},
			charCode: -1,
			highlightWhenActive: true,
			width: 1.5,
			textSize: 0.6,
			highlightAlt2: true,
			highlightAlt3: true,
			alt: true
		});
		
		add("<"); // 56023
		add(">"); // 57232
		
		add("Enter", {
			fun: function enter(click) {
				fireKey(13, "keydown");
			},
			charCode: 13,
			width: 2,
			textSize: 0.8
		});
		
		
		function add(char, options) {
			
			console.log("Adding virtual keyboard key: char=" + char + " orientation=" + orientation);
			
			if(options == undefined) options = {};
			
			var lowerCase = char.toLowerCase();
			var upperCase = char.toUpperCase();
			var charCode = lowerCase.charCodeAt(0);
			var charCodeCaps = upperCase.charCodeAt(0);
			var buttons = orientation == "horizontal" ? horizontalLayout : verticalLayout;
			
			if(options.charCode && !options.charCodeCaps) options.charCodeCaps = options.charCode;
			
			buttons.push({
				char: char, 
				charCode: options.charCode || charCode, 
				charCodeCaps: options.charCodeCaps || charCodeCaps, 
				row: options.row || row, 
				col: options.col || ++col,
				width: options.width || 1,
				textSize: options.textSize || 1,
				fun: options.fun || normalButton,
				alt1: options.alt1,
				alt2: options.alt2,
				alt3: options.alt3,
				highlightAlt1: options.highlightAlt1 || false,
				highlightAlt2: options.highlightAlt2 || false,
				highlightAlt3: options.highlightAlt3 || false,
				alt: options.alt || false // If it's an button that changes ALT1,ALT2
			});
		}
		
	}
	
	function removeButtons() {
		
	}
	
	
	
	function getMouseLocation(mouseEvent) {
		
		// Mouse position is on the current element (most likely Canvas)
		var mouseX = mouseEvent.offsetX==undefined?mouseEvent.layerX:mouseEvent.offsetX;
		var mouseY = mouseEvent.offsetY==undefined?mouseEvent.layerY:mouseEvent.offsetY;
		
		var badLocation = mouseX == undefined || mouseY == undefined || mouseX <= 0 || mouseY <= 0;
		
		if(mouseEvent.changedTouches && badLocation) {
			
			mouseX = Math.round(mouseEvent.changedTouches[mouseEvent.changedTouches.length-1].pageX); // pageX
			mouseY = Math.round(mouseEvent.changedTouches[mouseEvent.changedTouches.length-1].pageY);
			
			// The editor doesn't allow scrolling, so pageX is thus the same as clientX !
			
			// Touch events only have pageX which is the whole page. We only want the position on the canvas !?
			var rect = canvas.getBoundingClientRect();
			//console.log(rect.top, rect.right, rect.bottom, rect.left);
			mouseX = mouseX - rect.left;
			mouseY = mouseY - rect.top;
		}
		
		return {x: mouseX, y: mouseY};
	}
	
	
	function vrkeyboardMouseDown (mouseX, mouseY, caret, mouseDirection, button, target, keyboardCombo, mouseUpEvent) {
		return ALLOW_DEFAULT;
	}
	
	function vrkeyboardMouseUp(mouseX, mouseY, caret, mouseDirection, button, target, keyboardCombo, mouseUpEvent) {
		if(mouseUpEvent.type == "touchend") {
			
			if(useBuiltin) showBuiltinKeyboard();
			else if(useNative) showNativeKeyboard();
			else if(!usePhysical) {
				// Always show a keyboard after touch!
				if(lastUsedKeyboard == "builtin") showBuiltinKeyboard();
				else if(lastUsedKeyboard == "native") showNativeKeyboard();
				else throw new Error("Unknown lastUsedKeyboard=" + lastUsedKeyboard);
			}
			
			if(EDITOR.currentFile && !usePhysical) {
				// Wait for the resize, then scroll to the caret (where you clicked)
				setTimeout(function() {
					EDITOR.currentFile.scrollToCaret();
				}, 500);
			}
			
			// This function is only called when clicking on the file canvas
			
			// It seems the andorid physical keyboard wont enter characters after tabbing away and back ...
			// Can't seem to do anything about it :(
			
		}
		
		
		return ALLOW_DEFAULT;
	}
	
	
	
	
	// TEST-CODE-START
	// ### Test(s)
	
	function clickLetter(letter) {
		
		var id;
		
		if(buttons.length == 0) buttons = verticalLayout;
		
		for (var i=0; i<buttons.length; i++) {
			if(buttons[i].char==letter) {
				id = i;
				break;
			}
		}
		
		if(!id) throw new Error("Unable to find button " + letter + " buttons.length=" + (buttons && buttons.length));
		
		clickButton(id, event);
		
	}
	
	EDITOR.addTest(function testInsertAtCaret(callback) {
		// Make sure the characters are inserted in the right order.
		
		var input = document.createElement("input");
		
		var footer = document.getElementById("footer");
		
		footer.appendChild(input);
		EDITOR.resizeNeeded();
		
		
		// Seems we need an event to trigger the click event
		input.focus(); // Element needs to have focus *before* clicking on it for the click() event to trigger.
		//input.setAttribute("placeholder", "Double Click here to trigger the test!");
		input.setAttribute("id", "testVirtualKeyboardInput");
		
		
		//input.addEventListener("dblclick", function enterText() {
		
		clickLetter("a");
		clickLetter("b");
		clickLetter("c");
		
		setTimeout(function() {
			
			if(input.value != "abc") throw new Error("Unexpected input.value=" + input.value);
			
			footer.removeChild(input);
			
			return callback(true);
			
		}, 0);
		
		//});
		
	});
	
	// TEST-CODE-END
	
	
})();