
"use strict";

// Need to require non native modules here before we are chrooted

var module_iconv = require('iconv-lite');
var module_ftp = require('ftp');
var module_ssh2 = require('ssh2');

var UTIL = require("../client/UTIL.js");

// Optional modules:
try {
	var module_ps = require("ps-node");
}
catch(err) {
	console.log("Unable to load optional module(s): " + err.message);
}



var ftpQueue = []; // todo: Allow parrallel FTP commands (seems connection is dropped if you send a command while waiting for another)
var ftpBusy = false;

var API = {};

var DEFAULT_FILE_MODE = parseInt("0770", 8); // 1 = execute, 2 = write, 4 = read (note: umask is 0022)
var DEFAULT_FOLDER_MODE = parseInt("0770", 8); // 1 = execute, 2 = write, 4 = read (note: umask is 0022)
// Default is only the user are allowed to access files the user creates (umask 0022)

var FIND_FILES_ABORTED = false;
var FIND_FILES_IN_FLIGHT = 0;
var FIND_IN_FILES_ABORTED = false;
var ECHO_COUNTER = 0;

var EXEC_OPTIONS = {shell: "/bin/dash"};

var PROCESS = {}; // pid: spawned process

API.countLines = function countLines(user, json, callback) {

	API.readLines(user, json, function linesRead(err, json) {
		if(err) return callback(err);
		
		callback(null, {path: json.path, totalLines: json.totalLines});
		
	});
	
}

API.zip = function zipFolder(user, json, callback) {
	
	var folder = json.source;
	var destinationFolder = json.destination;
	var filename = json.filename;
	
	if(folder == undefined) return callback(new Error("No source specified!"));
	if(destinationFolder == undefined) destinationFolder = folder;
	if(filename == undefined) filename = UTIL.getFolderName(folder) + ".zip";
	
	var execFile = require('child_process').execFile;
	var execFileOptions = {cwd: folder, env: {HOME: "/", PATH:"/bin/:/usr/bin/"}};
	
	console.log("Creating zip archive: folder=" + folder + " destinationFolder=" + destinationFolder + " filename=" + filename);
	
	var exe = "/usr/bin/zip";
	var args =  ["-r", UTIL.joinPaths(destinationFolder, filename), folder, "--quiet"];
	
/*
todo: when zipping, first count the files in the folder,
then show a progress bar that is incremented for each stdout message from zip
*/

	execFile(exe, args, execFileOptions, function (err, stdout, stderr) {
		
		console.log(exe + " args=" + JSON.stringify(args) + " stderr=" + stderr + " stdout=" + stdout + " ");
		
		if(err) return callback(err);
		else if(stderr) return callback(stderr);
		else {
			
			return callback(null, {source: folder, destination: destinationFolder});
			
		}
	});
	
}

API.extract = function extract(user, json, callback) {
	
	var supportedFileTypes = ["zip", "rar", "gz", "tar.gz", "tgz"];
	
	var source = json.source;
	var destination = json.destination;
	
	if(source == undefined) callback(new Error("source: " + source));
	if(destination == undefined) destination = UTIL.getDirectoryFromPath(source) + UTIL.getFileNameWithoutExtension(source) + "/";
	
	var localSource = user.translatePath(source);
	if(localSource instanceof Error) return callback(localSource);
	
	var localDestination = user.translatePath(destination);
	if(localDestination instanceof Error) return callback(localDestination);
	localDestination = UTIL.trailingSlash(localDestination);
	
	var runningDirectory = UTIL.getDirectoryFromPath(localSource);
	
	var fileType = UTIL.getFileExtension(localSource);
	
	if(supportedFileTypes.indexOf(fileType) == -1) callback( new Error("File type (" + fileType + ") not supported! Try " + JSON.stringify(supportedFileTypes)) );
	
	if(fileType == "zip") {
var exe = "/usr/bin/unzip";
		//var exe = "/bin/echo";
		var args = [localSource, "-d", localDestination];
	}
	else if(fileType == "rar") {
		var exe = "/usr/bin/unrar";
		var args = ["e", localSource, localDestination];
	}
	else if(fileType == "gz") {
		// A gz file is only one file! The .gz part will be removed, and the unpacked file will replace the packed file.
		var exe = "/bin/gunzip";
		//var exe = "gunzip";
		var args = [localSource];
		destination = source.slice(0, -3);
	}
	else if(fileType == "tar.gz" || fileType == "tgz") {
		// It's a tarball that has been gzip'ed
		var exe = "/bin/tar";
		var args = ["zxvf", localSource, "-C " + localDestination];
	}
	else throw new Error("Unknown fileType=" + fileType);

	var execFile = require('child_process').execFile;
	var execFileOptions = {cwd: runningDirectory, env: {HOME: "/", PATH:"/bin/:/usr/bin/"}};
	
	console.log("user.homeDir=" + user.homeDir);
	
	execFile(exe, args, execFileOptions, function (err, stdout, stderr) {
		
		console.log(exe + " args=" + JSON.stringify(args) + " stderr=" + stderr + " stdout=" + stdout + " ");
		
		if(err) return callback(err);
		else if(stderr) return callback(stderr);
		else {
			
			if(exe == "/bin/gunzip" && json.destination) {
				// Move the file after gunzip'ing
				API.move(user, {oldPath: destination, newPath: json.destination}, function fileMoved(err, movedto) {
					if(err) return callback(new Error("gunzip succeeded, but move failed: " + err.message));
					else callback(null, {source: source, destination: movedto.path});
				});
			}
			else return callback(null, {source: source, destination: destination});
			
		}
	});
}

API.hash = function hash(user, json, callback) {
	// Useful for example comparing files, so that files don't need to be uploaded to the client for comparison.
	
	var crypto = require('crypto');
	var hash = crypto.createHash('sha256');
	
	if(!json.path && json.text) {
		// Hash the text (not the file)
		hash.update(json.text);
		callback(null, hash.digest('hex'));
		callback = null;
		return;
}
	
	var path = user.translatePath(json.path);
	if(path instanceof Error) return callback(path);
	
	// Check path for protocol
	var url = require("url");
	var parse = url.parse(path);
	
	var input; // Read stream
	
	if(parse.protocol == "ftp:" || parse.protocol == "ftps:") {
		if(user.remoteConnections.hasOwnProperty(parse.hostname)) {
			var c = user.remoteConnections[parse.hostname].client;
			console.log("Getting file hash from FTP server: " + parse.pathname);
			c.get(parse.pathname, function getFtpFileStream(err, fileReadStream) {
				if(err) throw err;
				input = fileReadStream;
				input.on("readable", readStream);
				input.on('error', streamError);
});
}
else {
			callback(new Error("No connection open to FTP on " + parse.hostname + " !"));
		}
	}
else if(parse.protocol == "sftp:") {
		if(user.remoteConnections.hasOwnProperty(parse.hostname)) {
			var c = user.remoteConnections[parse.hostname].client;
			console.log("Getting file hash from SFTP server: " + parse.pathname);
			
			input = c.createReadStream(parse.pathname);
			input.on("readable", readStream);
			input.on('error', streamError);
			
		}
		else {
			callback(new Error("No connection open to SFTP on " + parse.hostname + " !"));
		}
	}
	else {
		// Asume local file system
		var fs = require('fs');
		
		input = fs.createReadStream(path);
		input.on('readable', readStream);
		input.on('error', streamError);
	}
	
	function readStream() {
		//console.log("Stream readable! path=" + path);
		var data = input.read();
		//console.log("Stream data: " + data);
		if (data)
			hash.update(data);
		else {
			if(callback) {
				callback(null, hash.digest('hex'));
				callback = null;
			}
		}
	}
	
	function streamError(err){ 
		//console.log("Stream error! path=" + path);
		var error = new Error("Unable to hash file: " + path + "\n " + err.message);
		error.code = err.code;
		callback(error);
		callback = null;
	}
}

API.httpGet = function httpGet(user, options, callback) {
	var url = options.url;
	
	if(url == undefined) return callback(new Error("URL is needed!"));
	
	var loc = UTIL.getLocation(url);
	
	if(loc.protocol == "http") {
		var reqModule = require("http");
	}
	else if(loc.protocol == "https") {
		var reqModule = require("https");
	}
	else {
		return callback(new Error("Unsupported protocol: " + loc.protocol + " in url=" + url));
	}
	
	var req;
	var gotError = null;
	var redirects = 0;
	var maxRedirects = 10;
	
	makeReq(url);
	
	function makeReq(url) {
		req = reqModule.request(url, gotResp);
		
		req.on("error", function(err) {
			callback(err);
			callback = null;
			gotError = true;
		});
		
		req.end();
	}
	
	function gotResp(resp) {
		//resp.setEncoding('utf8');
		
		if(resp.statusCode == 302 && resp.headers.location) {
			redirects++;
			if(++redirects > maxRedirects) {
callback(new Error("Too many redirects! redirects=" + redirects));
				callback = null;
				return;
			}
			makeReq(resp.headers.location);
			return;
		}
		
		var data = [];
		
		resp.on('data', respData);
		resp.on('end', respEnd);
		
		function respData(chunk) {
			data.push(chunk);
		}
		
		function respEnd() {
			if(gotError) return;
			
			
			var buffer = Buffer.concat(data);
			
			if(options.binary) {
				callback(null, buffer);
				callback = null;
				return;
			}
			
			var body = buffer.toString('utf8');
			
			// Detect encoding
			console.log("Headers: " + JSON.stringify(resp.headers));
			
			if(resp.headers.hasOwnProperty("content-type")) {
				// Ex: Content-Type: text/html; charset=utf-8
				var matchCharset = resp.headers["content-type"].match(/charset=([^;]*)/i);
				if(matchCharset) {
					var charset = matchCharset[1].trim().toLowerCase();
				}
			}
			
			if(!charset) {
				var matchCharset = body.match(/charset\s*=\s*["']([^'"]*)["']/i); // ex:  <meta charset="UTF-8">
				if(matchCharset) {
					var charset = matchCharset[1].trim().toLowerCase();
				}
			}
			
			if(!charset) {
				var matchCharset = body.match(/charset=([^;'"]*)/i); // ex:  <meta http-equiv="Content-Type" content="text/html; charset=utf-8" />
				if(matchCharset) {
					var charset = matchCharset[1].trim().toLowerCase();
				}
			}
			
			if(charset && !(charset == "utf-8" || charset == "utf8")) {
				console.log("Detected charset=" + charset);
				
				console.log("iconv.encodingExists('" + charset + "')=" + module_iconv.encodingExists(charset));
				
				if(!module_iconv.encodingExists(charset)) {
					gotError = true;
					callback(new Error("Unable to decode charset=" + charset));
					callback = null;
					return;
				}
				else {
					body = module_iconv.decode(buffer, charset);
				}
			}
			callback(null, body);
			callback = null;
		}
	}
}

API.download = function download(user, json, callback) {
	
	console.log("download: " + JSON.stringify(json));
	
	var binary = true;
	
	if(json.type == "text") {
		binary = false;
	}
	
	var downloadResp = {};
	
	API.httpGet(user, {url: json.url, binary: binary}, function gotHttpData(err, buffer) {
		if(err) return callback(new Error("Failed to request URL " + json.url + " : " + err.message));
		
		console.log("buffer.length=" + buffer.length);
		
		if(json.createPath) {
			var parentFolder = UTIL.getDirectoryFromPath(json.path);
			API.createPath(user, {pathToCreate: parentFolder}, function pathCreated(err) {
				if(err) return callback(new Error("Problem creating path to folder " + parentFolder + ": " + err.message)); 
				else save();
			});
		}
		else save();
		
		function save() {
			var saveOptions = {path: json.path, text: buffer}
			
			if(binary) {
				saveOptions.inputBuffer = true;
			}
			else {
				downloadResp.text = buffer;
			}
			
			API.saveToDisk(user, saveOptions, function saved(err, saveResp) {
				if(err) return callback(new Error("Failed to save file to disk " + json.path + ": " + err.message));
				
				for(var key in saveResp) downloadResp[key] = saveResp[key];
				
				callback(err, downloadResp);
			});
		}
	});
}

API.readLines = function readLines(user, json, callback) {
	/*
		
		note: Don't specify lineBreak unless you know the line-break convention, or the result might end up with one big line!
		
	*/
	
	//console.log("readLines: json=" + JSON.stringify(json)); 
	
	if(json.path == undefined) return callback(new Error("No path property in options: " + JSON.stringify(json)));
	
	var path = user.translatePath(json.path);
	if(path instanceof Error) return callback(path);
	
	var url = require("url");
	var parse = url.parse(path);
	
	var encoding = json.encoding || "utf8";
	var lb = json.lineBreak || undefined;
	var startLine = json.start || 1;
	var MAX_LINES = json.max || 10000;
	var endLine = json.end || MAX_LINES;
	var lines = [];
	var stream;
	var text = "";
	var totalLines = 0;
	var chunkSize = json.chunkSize; // Useful when testing
	
	var doneReading = false;
	var readWhenReady = true;
	var text = "";
	var textHead = "";
	var StringDecoder = require('string_decoder').StringDecoder;
	var decoder = new StringDecoder(encoding);
	var rowsWanted = (endLine-startLine) + 1;
	
	var readOptions = {};
	if(chunkSize) readOptions.highWaterMark = chunkSize;
	
	if(!callback) {
		throw new Error("No callback defined!");
	}
	
	if(startLine < 1) return callback("start line can not be below line 1 (line 1 is the first line)");
	if(startLine > endLine) return callback("start line can not be above end line!");
	
	// todo: Support ftp/ftps and ftps !!?
	
	if(parse.protocol == "ftp:" || parse.protocol == "ftps:") {
		
		return callback(new Error("readLines on ftp or ftps not yet supported!"));
		
		if(user.remoteConnections.hasOwnProperty(parse.hostname)) {
			
			var c = user.remoteConnections[parse.hostname].client;
			
			console.log("Getting file from FTP server: " + parse.pathname);
			
			c.get(parse.pathname, function getFtpFileStream(err, fileReadStream) {
				
				if(err) throw err;
				
				console.log("Reading file from FTP ...");
				
				console.log(fileReadStream);
				
				stream = fileReadStream;
				
				stream.setEncoding('utf8');
				stream.on('readable', streamReadable);
				stream.on("end", streamEnded);
				stream.on("error", streamError);
				stream.on("close", streamClose);
				
				// Hmm it seems the FTP module uses "old" streams:
				var StringDecoder = require('string_decoder').StringDecoder;
				var decoder = new StringDecoder('utf8');
				var str;
				stream.on('data', function(data) {
					str = decoder.write(data);
						fileContent += str;
						console.log('loaded part of the file');
					});
					
				});
				
			}
			else {
			user.send({msg: "No connection open to FTP on " + parse.hostname + " !"});
			}
		}
		else if(parse.protocol == "sftp:") {
			
		return callback(new Error("readLines on sftp not yet supported!"));
		
			if(user.remoteConnections.hasOwnProperty(parse.hostname)) {
				
				var c = user.remoteConnections[parse.hostname].client;
				
				console.log("Getting file from SFTP server: " + parse.pathname);
				
				var options = {
				encoding: encoding
				}
				// Could also use sftp.createReadStream
				c.readFile(parse.pathname, options, function getSftpFile(err, buffer) {
					
					if(err) console.warn(err.message);
					
					callback(err, {path: path, data: buffer.toString("utf8")});
					
					
				});
				
			}
			else {
			user.send({msg: "No connection open to SFTP on " + parse.hostname + " !"});
			}
		}
		
		else {
			
			// Assume local file system
			var fs = require("fs");
			if(path.indexOf("file://") == 0) path = path.substr(7); // Remove file://
			
		stream = fs.createReadStream(path, readOptions);
		stream.on('readable', streamReadable);
			stream.on("end", streamEnded);
			stream.on("error", streamError);
			stream.on("close", streamClose);
			
		}
		
		
		// Functions to handle NodeJS ReadableStream's
		function streamClose() {
			//console.log("Stream closed! path=" + path);
		
		if(callback) callback(null, {path: path, lines: lines, end: Math.min(endLine, totalLines), totalLines: totalLines, lineBreak: lb});
		callback = null;
		}
		
		function streamError(err) {
			console.log("Stream error! path=" + path);
			if(callback) callback(err);
			callback = null;
		}
		
		function streamEnded() {
		//console.log("Stream ended! path=" + path + " lines.length=" + lines.length + " totalLines=" + totalLines + " startLine=" + startLine + " endLine=" + endLine + " textHead.length=" + textHead.length + " text.length=" + text.length );
		doneReading = true;
		
		if(text) readRows();
		}
	
	function streamReadable() {
		// The 'readable' event is emitted when there is data available to be read from the stream
		// note: It will be called many times!
		//console.log("stream readable!");
		if(readWhenReady) read();
	}
		
	
	function read() {
		// Read from the stream
		readWhenReady = false;
		
		if(textHead) {
			text = textHead + text;
			textHead = "";
		}
		
		var chunk; // note: chunk is not a string! use decoder!
		
		// Use a while loop instead of recursively calling read to avoid stack limits
		while(chunk = stream.read()) {
			
			// chunk is Not a string! And it can cut utf8 characters in the middle, so use decoder
			text += decoder.write(chunk);
			
			//console.log("text=" + UTIL.lbChars(text));
			//console.log("text.length=" + text.length);
			
			if(!lb) lb = UTIL.determineLineBreakCharacters(text);
			
			// Don't remove any line breaks here! Doing so might concatenate two rows!
			
			if(text.indexOf(lb) == -1) {
				console.log("Text does not contain a line break. Continue reading ...");
				continue;
			}
			
			if(text.slice(text.length-lb.length) != lb) {
				textHead = text.slice(text.lastIndexOf(lb)+1); // Will be the start of the text at next read
				text = text.slice(0, text.lastIndexOf(lb)); // Last lb not included
				console.log("textHead.length=" + textHead.length + " text.length=" + text.length);
			}
			else {
				text = text.slice(0, -lb.length); // Remove the ending lb
			}
			
			// As the ending line-break was removed above - one single linebreak actually means two empty rows!
			
			//console.log("Read " + chunk.length + " bytes from " + path);
			
			readRows();
			
		}
		
		if(!doneReading) {
			//console.warn("chunk=" + chunk + " but doneReading=" + doneReading);
			readWhenReady = true;
			//console.log("Waiting for stream readable ...");
		}
	}
	
	function readRows() {
		
		// Fix inconsistent line breaks
		if(lb == "\n") {
			text = text.replace(/\r/g, ""); // Just remove all CR
		}
		else if(lb == "\r\n") {
			// Look for lonely LF
			var lfIndex = -1;
			while(true) {
				lfIndex = text.indexOf("\n", lfIndex+1);
				if(lfIndex == -1) break;
				if(text.charAt(lfIndex-1) != "\r") {
					// Remove lonely LF
					text = text.slice(0, lfIndex) + text.slice(lfIndex+1);
					--lfIndex;
				}
			}
		}
		
		var rows = text.split(lb);
		
		totalLines += rows.length;
		
		//console.log("rows.length=" + rows.length + " lb=" + UTIL.lbChars(lb) + " text.length=" + text.length);
		
		text = "";
		
		// Where on the rows to start in case endLine > total available rows
		var start = startLine - lines.length - totalLines + rows.length - 1;
		
		var rowsToAdd = rowsWanted - lines.length;
		
		//console.log("rows.length=" + rows.length + " totalLines=" + totalLines + " startLine=" + startLine + " endLine=" + endLine + " lines.length=" + lines.length + " rowsWanted=" + rowsWanted + " start=" + start + " rowsToAdd=" + rowsToAdd);
		
		//console.log(rows);
		
		if(totalLines >= startLine && lines.length < rowsWanted) {
			if( rowsToAdd < rows.length || start > 0) rows = rows.splice(start, rowsToAdd);
			lines = lines.concat(rows);
			//console.log(" Added " + rows.length + " rows.");
			//console.log(rows);
		}
	}
}


API.writeLines = function writeLines(user, json, writeLinesCallback) {
	/*
		
		start: Start line
		end: End line
		content: The content to write
		overwrite: If the current text between start and end line should be overwritten with the new content
		If overwrite is set to false or undefined the content will be added at the start line
		
	*/
	
	var start = json.start;
	var end = json.end;
	var content = json.content;
	var overwrite = json.overwrite;
	var path = user.translatePath(json.path);
	if(path instanceof Error) return callback(path);
	
	if(overwrite != undefined && end == undefined) return writeLinesCallback(new Error("option overwrite=" + overwrite + " but end=" + end + " "));
	if(overwrite == undefined && end != undefined) return writeLinesCallback(new Error("Expected overwrite=" + overwrite + " to be true when end=" + end + " is set!"));
	
var encoding = "utf8";
	var fs = require("fs");
	var tmpPath = path + ".tmp";
	var lb = UTIL.determineLineBreakCharacters(content);
	
	if(content.slice(content.length-lb.length) != lb) {
		console.log("writeLines: content.length=" + content.length + " did not end with a " + UTIL.lbChars(lb) + " line break!");
	}
	else if(content.indexOf(lb) != -1) {
		content = content.slice(0, -lb.length); // Remove the ending lb
	}
	// Expect all lines to end with a line break.
	// But do not include the last line break, so that a lb can be appended after all contentRows 
	var contentRows = content.split(lb);
	var totalRowsWritten = 0;
	var totalRowsRead = 0;
	
	var chunkSize = json.chunkSize; // Useful when testing
	
	//console.log("writeLines: start=" + start + " end=" + end + " overwrite=" + overwrite + " path=" + path + " lb=" + UTIL.lbChars(lb) + " content.length=" + content.length + "  ");
	
	if(!UTIL.isLocalPath(path)) return writeLinesCallback(new Error("writeLines currently only supports local files!"));
	if(overwrite && !end) return writeLinesCallback(new Error("end line need to be specified if overwriting!"));
	
	var StringDecoder = require('string_decoder').StringDecoder;
	var decoder = new StringDecoder(encoding);
	var text = "";
	var textHead = "";
	var doneReading = false;
	var line = 1;
	var isWriting = false;
	var contentWritten = false;
	var tmpClosed = false;
	var originalClosed = false;
	var finished = false;
	var hasStarted = false;
	var fileEndsWithLineBreak = false;
	var readWhenReady = false;
	var lastWrite = false; // Set to true before calling write for the last time
	
	var readOptions = {};
	if(chunkSize) readOptions.highWaterMark = chunkSize;
	var original = fs.createReadStream(path, readOptions);
	var originalReadable = false;
	original.on('readable', function() {
		// The 'readable' event is emitted when there is data available to be read from the stream
		// note: It will be called many times!
		//console.log("writeLines: original:readable: Read stream now readable!");
		//if(originalReadable) console.warn("read stream readable called twice!");
		originalReadable = true;
		if(tmpReady && !hasStarted) begin();
		else if(!hasStarted) console.log("Waiting for write stream ready ... tmpReady=" + tmpReady + " hasStarted=" + hasStarted);
		else if(readWhenReady) read();
});
	original.on("end", function() {
		// The 'end' event is emitted when there is no more data to be consumed from the stream.
		//console.log("writeLines: original:end: Read stream ended! textHead.length=" + textHead.length + " text.length=" + text.length + " isWriting=" + isWriting);
		
		doneReading = true;

		// Call read one last time to make sure everything get written
		read();
	});
	original.on("error", function(err) {
		console.log("writeLines: original:error: " + err.message);
		finished = true;
		writeLinesCallback(new Error("Problem with read stream: " + err.message));
	});
	original.on("close", function() {
		// The 'close' event is emitted when the stream and any of its underlying resources (a file descriptor, for example) have been closed. The event indicates that no more events will be emitted, and no further computation will occur.
		console.log("writeLines: original:close: doneReading=" + doneReading + " contentWritten=" + contentWritten);
		if(originalClosed) console.warn("writeLines: original:close: read stream close called twice!");
		originalClosed = true;
		if(tmpClosed && !finished) finish();
		else console.log("writeLines: original:close: Waiting for write stream to close ...");
	});
	
	var writeOptions = {};
	if(chunkSize) writeOptions.highWaterMark = chunkSize;
	console.log("writeLines: Creating write stream tmpPath=" + tmpPath + " writeOptions=" + JSON.stringify(writeOptions));
	var tmp = fs.createWriteStream(tmpPath, writeOptions);
	var tmpReady = false;
	tmp.on('ready', function() {
		// Emitted when the fs.WriteStream is ready to be used.
		console.log("writeLines: write stream ready!");
		if(tmpReady) console.warn("writeLines: write stream ready called twice!");
		tmpReady = true;
		if(originalReadable && !hasStarted) begin();
		else if(!hasStarted) console.log("writeLines: Waiting for read stream readable ...");
	});
	tmp.on("error", function(rtt) {
		console.log("writeLines: tmp stream error: " + err.message);
		finished = true;
		writeLinesCallback(new Error("Problem with write stream: " + err.message));
	});
	tmp.on("close", function() {
		// Emitted when the WriteStream's underlying file descriptor has been closed.
		console.log("writeLines: tmp stream closed! doneReading=" + doneReading + " contentWritten=" + contentWritten);
		if(tmpClosed) console.warn("writeLines: write stream close called twice!");
		tmpClosed = true;
		if(originalClosed && !finished) finish();
		else console.log("writeLines: Waiting for read stream to close");
	});
	
	function finish() {
		console.log("writeLines: finish!");
		if(finished) throw new Error("finished=" + finished + " tmpClosed=" + tmpClosed + " originalClosed=" + originalClosed + ". Close called twice !?");
		finished = true;
		
		// We should now have a .tmp file containing the original file with the inserted content
		fs.stat(tmpPath, function(err, stats) {
			if(err) return writeLinesCallback(new Error("Unable to stat tmpPath=" + tmpPath + " Error: " + err.message));
			
			if(stats.size == 0) return writeLinesCallback(new Error("tmpPath=" + tmpPath + " stats.size=" + stats.size));
			
			// Remove the original file
			fs.unlink(path, function(err) {
				if(err) return writeLinesCallback(new Error("Failed to remove original file: path=" + path + ""));
				
				// Rename the tmp file to the original
				fs.rename(tmpPath, path, function(err) {
					if(err) return writeLinesCallback(new Error("Failed to rename tmpPath=" + tmpPath + " to path=" + path));
					else writeLinesCallback(null, {totalRowsWritten: totalRowsWritten, totalRowsRead: totalRowsRead, contentRows: contentRows.length});
				});
				
			});
			
		});
	}
	
	function begin() {
		console.log("writeLines: begin!");
		if(hasStarted) throw new Error("begin() called twice!");
		hasStarted = true;
		read();
	}
	
	function read() {
		
		//console.log("writeLines: read: textHead=" + UTIL.lbChars(UTIL.shortString(textHead)) + " text=" + UTIL.lbChars(UTIL.shortString(text)) + " ");
		
		if(isWriting) {
			console.log("writeLines: read: Waiting for write to be done ...");
			return;
		}
		
		readWhenReady = false;
		
		if(textHead) {
			text = textHead + text;
			textHead = "";
		}
		
		var chunk = original.read();
		
		if(chunk == null) {
			//console.log("writeLines: read: chunk=" + chunk + " text.length=" + text.length + " textHead.length=" + textHead.length + " doneReading=" + doneReading + " contentWritten=" + contentWritten + " isWriting=" + isWriting);
			
			// This has a probability to happen *before* the read stream end event!
			
			if(!doneReading) {
				readWhenReady = true;
				//console.log("writeLines: read: Waiting for readable ... doneReading=" + doneReading);
				return;
			}
			
			if(text.length == 0 && contentWritten && !isWriting) {
				console.log("writeLines: read: Ending write stream because there's nothing more to write! doneReading=" + doneReading + " contentWritten=" + contentWritten + " isWriting=" + isWriting);
				tmp.end(function() {
					console.log("writeLines: read: Write stream ended! isWriting=" + isWriting);
				});
				return;
			}
		}
		else {
		// chunk is Not a string! And it can cut utf8 characters in the middle, so use decoder
		text += decoder.write(chunk);
		}
		console.log("writeLines: read: text=" + UTIL.lbChars(UTIL.shortString(text)));
		
		// Don't remove any line breaks here! Doing so might concatenate two rows!
		
		if(!doneReading) {
			if(text.indexOf(lb) == -1) {
				console.log("writeLines: read: Text does not contain a line break. Continue reading ...");
			read();
			return;
		}
		
			if(text.slice(text.length-lb.length) != lb) {
				textHead = text.slice(text.lastIndexOf(lb)+lb.length); // Will be the start of the text at next read
				text = text.slice(0, text.lastIndexOf(lb)); // Set the text to everything up until but not including the last line break
				console.log("writeLines: read: textHead.length=" + textHead.length + " text.length=" + text.length);
				console.log("writeLines: read: Sliced textHead=" + UTIL.lbChars(UTIL.shortString(textHead)) + " text=" + UTIL.lbChars(UTIL.shortString(text)) + " ");
		}
			else {
			text = text.slice(0, -lb.length); // Remove the ending lb
		}
		}
		
		// text.length==0 means the text had a line break, but it has been removed!
		// So continue and insert a empty line!
		
		
		var rows = text.split(lb);
		// As the ending line-break was removed above, one single linebreak actually means two empty rows!
		
		text = ""; // Reset the text, that has been converted into rows, that will now be inserted
		
		processRows(rows, read);
		
		//console.log("writeLines: read: line=" + line + " doneReading=" + doneReading + " rows.length=" + rows.length + " Read " + (chunk && chunk.length) + " bytes from " + path);
		
	}
	
	function processRows(rows, callback) {
		//console.log("rows=" + JSON.stringify(rows));
		
		console.log("writeLines: processRows: Line " + line + ": " + rows[0]);
		
		console.log("writeLines: processRows: totalRowsRead=" + totalRowsRead);
		
		totalRowsRead += rows.length;
		
		/*
			Cut off head and tail off the rows depending on where in the stream we are
			
			If we are not over-writing we only have to worry about where to insert the content.
			
			head: Everything up to but not including start. The part to be written before writing the content
			tail: Everything after end including start, or everything and not including end The part the be written after the content
			rows: Remaining rows are ignored
			
		*/
		
		var headIndex = 0; // head always start at zero
		var headLength = start-line;
		
		if(end) {
			var tailIndex = end-line +1;
		}
		else {
			var tailIndex = start-line;
		}
		
		var tailLength = rows.length - tailIndex + 1;
		
		console.log("writeLines: processRows: rows.length=" + rows.length + " line=" + line + " start=" + start + " headIndex=" + headIndex + " headLength=" + headLength +
		" end=" + end + " tailIndex=" + tailIndex + " tailLength=" + tailLength + "");
		
		line += rows.length;
		
		// Cut off the tail first to make the calculation above easier
		if(tailIndex > -1) var tail = rows.splice(tailIndex, tailLength);
		else var tail = rows.splice(0, rows.length);
		
		var head = rows.splice(headIndex, headLength);
		
		console.log("writeLines: processRows: head.length=" + head.length + " tail.length=" + tail.length + " rows.length=" + rows.length + " overwrite=" + overwrite + "");
		
		//console.log("writeLines: processRows: head=" + JSON.stringify(head));
		//console.log("writeLines: processRows: tail=" + JSON.stringify(tail));
		//console.log("writeLines: processRows: rows=" + JSON.stringify(rows));
		
		//if(tail.length == 0 && rows.length > 0 && !overwrite) tail = rows;
		
		if(rows.length > 0 && !overwrite) throw new Error("Unexpected rows length! rows.length=" + rows.length + " overwrite=" + overwrite + " head.length=" + head.length + " tail.length=" + tail.length);
		
		write(head, function() {
			if(line < start) {
				if(tail.length > 0) throw new Error("Unexpected tail length! rows.length=" + rows.length + " overwrite=" + overwrite + " head.length=" + head.length + " tail.length=" + tail.length);
				callback();
			}
			else if(!contentWritten) {
				write(contentRows, function() {
					write(tail, callback);
				});
				contentWritten = true;
			}
			else write(tail, callback);
		});
	}
	
	function write(rows, callback) {
		if(isWriting) console.warn("writeLines: write: Write in progress!");
		
		//console.log("writeLines: write: rows.length=" + rows.length + " : 0=" + rows[0]);
		
		isWriting = true;
		var row = 0;
		var rowsToWrite = rows.length;
		
		//console.log("writeLines: write: " + JSON.stringify(rows));
		
		if(rows.length == 0) {
			//console.warn("writeLines: write: Zero rows!");
			return done();
		}
		
		writeRow();
		
		function done() {
			isWriting = false;
			//console.log("writeLines: write:done: Last row of " + rowsToWrite + " rows written. totalRowsWritten=" + totalRowsWritten);
			callback();
		}
		
		function writeRow() {
			
			var ok = true;
			do {
				if (row == rows.length-1) {
					// last write and last row
					// Always write a line break after each row! 
					// Or the read logic would become very complicated. For example when the chunk stops right before a lb.
					// All rows need to end with a lb!
					tmp.write(rows[row] + lb, encoding, done);
				}
				else {
					// see if we should continue, or wait
					// don't pass the callback, because we're not done yet.
					ok = tmp.write(rows[row] + lb, encoding);
				}
				row++;
				totalRowsWritten++;
			} while (row < rows.length && ok);
				
			
			if (row < rows.length) {
				// had to stop early!
				// write some more once it drains
				console.log("writeLines: writeRow: Waiting for drain ...");
				tmp.once('drain', writeRow);
			}
		}
	}
	}




API.readFromDisk = function readFromDisk(user, json, callback) {
	
	var returnBuffer = json.returnBuffer;
	var encoding = json.encoding;
	var path = json.path;
	var fileContent = "";
	var stream;
	var fileBuffer = [];
	
	if(path == undefined) return callback(new Error("No path specified in options to API.readFromDisk!"));
	
	if(!callback) {
		throw new Error("No callback defined!");
	}
	
	var fs = require("fs");
	var crypto = require('crypto');
	
	var shasum = crypto.createHash('sha256');
	
	console.log("Reading file from disk: " + path + " returnBuffer=" + returnBuffer + " encoding=" + encoding);
	//console.log(UTIL.getStack("Read from disk"));
	
	// Check path for protocol
	var url = require("url");
	var parse = url.parse(path);
	
	if(parse.protocol == "ftp:" || parse.protocol == "ftps:") {
		
		if(user.remoteConnections.hasOwnProperty(parse.hostname)) {
			
			var c = user.remoteConnections[parse.hostname].client;
			
			console.log("Getting file from FTP server: " + parse.pathname);
			
			c.get(parse.pathname, function getFtpFileStream(err, fileReadStream) {
				
				if(err) return callback(err);
				
				console.log("Reading file from FTP ...");
					
					console.log(fileReadStream);
					
					stream = fileReadStream;
					
					stream.setEncoding('utf8');
					stream.on('readable', readStream);
					stream.on("end", streamEnded);
					stream.on("error", streamError);
					stream.on("close", streamClose);
					
					// Hmm it seems the FTP module uses "old" streams:
					var StringDecoder = require('string_decoder').StringDecoder;
					var decoder = new StringDecoder('utf8');
					var str;
					stream.on('data', function(data) {
					if(returnBuffer) fileBuffer.push(data);
					else {
str = decoder.write(data);
						shasum.update(data);
						fileContent += str;
					}
					console.log('loaded part of the file');
					});
					
				});
				
			}
			else {
			user.send({msg: "No connection open to FTP on " + parse.hostname + " !"});
			}
		}
		else if(parse.protocol == "sftp:") {
			
			if(user.remoteConnections.hasOwnProperty(parse.hostname)) {
				
				var c = user.remoteConnections[parse.hostname].client;
				
				console.log("Getting file from SFTP server: " + parse.pathname);
				
			var options = {};
			// Having encoding in options encodes the buffer,
			// So if we want the buffer we should not set the encoding option!
			if(!returnBuffer) options.encoding = encoding || "utf8";
			
				// Could also use sftp.createReadStream
				c.readFile(parse.pathname, options, function getSftpFile(err, buffer) {
					var resp = {path: path};
				
					if(err) {
					
					console.warn(err.message);
					
					if(err.message == "No such file") {
						err = new Error(err.message + " (err.code=" + err.code + ")");
						err.code = "ENOENT";
					}
					
				}
				else {
				
					console.log("getSftpFile: returnBuffer=" + returnBuffer + " typeof buffer=" + (typeof buffer) + " isBuffer=" + Buffer.isBuffer(buffer));
					
					if(returnBuffer) {
						resp.data = buffer;
						shasum.update(buffer);
					}
					else {
						resp.data = buffer.toString("utf8");
						shasum.update(resp.data);
					}
					
					resp.hash = shasum.digest('hex');
				}
				
					callback(err, resp);
					});
				
			}
			else {
			user.send({msg: "No connection open to SFTP on " + parse.hostname + " !"});
			}
		}
		
		else {
		
			// Asume local file system
			
		var module_path = require("path");
		if(!module_path.isAbsolute(path)) {
			var error = new Error("Not an absolute path: " + path);
			error.code = "NOT_ABSOLUTE";
			return callback(error);
		}
		
		var path = user.translatePath(json.path);
		if(path instanceof Error) return callback(path);
		
			if(path.indexOf("file://") == 0) path = path.substr(7); // Remove file://
			
		// If no encoding is specified in fs.readFile, then the raw buffer is returned.
				
		console.log("Read from disk: path=" + path);
		
				fs.readFile(path, function(err, buffer) {
			if(err) return callback(err);
			else {
			//shasum.update(buffer.toString(encoding));
			shasum.update(buffer); // Doesn't seem to matter if you pass it buffer or utf8 string!
				
				if(encoding == undefined) encoding = "utf8";
			
			callback(err, {
				path: user.toVirtualPath(path), 
				data: returnBuffer ? buffer : buffer.toString(encoding), 
				hash: shasum.digest('hex')
			});
			}
				});
			
		/*
			if(encoding == undefined) encoding = "utf8";
				fs.readFile(path, encoding, function(err, string) {
					if(err) console.warn(err.message);
					callback(err, {path: user.toVirtualPath(path), data: string});
					});
		*/
		
		}
		
		// Functions to handle NodeJS ReadableStream's
		function streamClose() {
			console.log("Stream closed! path=" + path);
		}
		
		function streamError(err) {
			console.log("Stream error! path=" + path);
			throw err;
		}
		
		function streamEnded() {
			console.log("Stream ended! path=" + path);
			
		var resp = {path: path};

		resp.hash = shasum.digest('hex');
		
		if(returnBuffer) {
			resp.data = fileBuffer;
}
		else {
			resp.data = fileContent;
		}
		
		callback(null, resp);
			
		}
	
		function readStream() {
			// Called each time there is something comming down the stream
			
			var chunk;
			var str = "";
			var StringDecoder = require('string_decoder').StringDecoder;
			var decoder = new StringDecoder('utf8');
			
			//var chunkSize = 512; // How many bytes to recive in each chunk
			
			console.log("Reading stream ... isPaused=" + stream.isPaused());
			
			while (null !== (chunk = stream.read()) && !stream.isPaused() ) {
				
			shasum.update(chunk);
			
				// chunk is Not a string! And it can cut utf8 characters in the middle, so use decoder
				str = decoder.write(chunk);
				
				fileContent += str;
				
				console.log("Got chunk! str.length=" + str.length + "");
				
			}
		}
	}

API.copyFile = function copyFile(user, json, callback) {
	
	var source =json.from;
	var target = json.to;
	
	// Both API.readFromDisk and API.saveToDisk will translate the path, so we don't have to run user.translatePath() here!
	
	// Use buffers and Not text, or images will not work!
	API.readFromDisk(user, {path: source, returnBuffer: true}, function fileRead(err, read) {

		if(err) return callback(err);
		
		API.saveToDisk(user, {path: target, text: read.data, inputBuffer: true, public: json.public}, function fileWrite(err, write) {
			
			if(err) return callback(err);
			else callback(null, {to: write.path});
			
		});
		
	});
	
	/*
		var options = {
		mode:  DEFAULT_FILE_MODE
		};
		
		if(json.public) {
		// Make it so everyone can read it
		options.mode = parseInt("0777", 8);
		// note: The file permissions wont change if the file already exists!
		}
		
	var cbCalled = false;
	
	var fs = require("fs");
	
	var rd = fs.createReadStream(source);
	rd.on("error", function(err) {
		done(err);
	});
	
	var wr = fs.createWriteStream(target, options);
	wr.on("error", function(err) {
		done(err);
	});
	wr.on("close", function(ex) {
		done();
	});
	rd.pipe(wr);

	function done(err) {
		if (!cbCalled) {
			
			callback(err, {to: target});
			cbCalled = true;
		}
	}
	*/
	
}

API.move = function move(user, json, callback) {
	/*
		
		note: Use EDITOR.move ! (don't call this directly)
		
	*/
	
	var oldPath = json.oldPath;
	var newPath = json.newPath;
	
	if(oldPath == undefined) return callback(new Error("oldPath=" + oldPath + " can not be null or undefined!"));
	if(newPath == undefined) return callback(new Error("newPath=" + newPath + " can not be null or undefined!"));
	
	oldPath = user.translatePath(oldPath);
	if(oldPath instanceof Error) return callback(oldPath);
	
	newPath = user.translatePath(newPath);
	if(newPath instanceof Error) return callback(newPath);
	
	// Figure out if it's a directory or a file
	var lastChar = oldPath.charAt(oldPath.length-1);
	if(lastChar == "/" || lastChar == "\\") {
		// It's a directory!'
		// Figure out protocol
		var url = require("url");
		var parse = url.parse(oldPath);
		var dest = url.parse(oldPath);

if(parse.protocol && parse.hostname != dest.hostname) {
return callback(new Error("Moving folders between servers not yet implemented!"));
}

		if(parse.protocol == "ftp:" || parse.protocol == "ftps:") {
			if(user.remoteConnections.hasOwnProperty(parse.hostname)) {
				var c = user.remoteConnections[parse.hostname].client;
				c.rename(parse.pathname, dest.pathname, function renamedFileOnFtp(err) {
					callback(err);
});
			}
			else {
				callback(new Error("Failed to move: " + oldPath + "\nNo connection open to FTP on " + parse.hostname + " !"));
			}
		}
		else if(parse.protocol == "sftp:") {
			if(user.remoteConnections.hasOwnProperty(parse.hostname)) {
				var c = user.remoteConnections[parse.hostname].client;
				console.log("Attempting to rename folder " + parse.pathname + " to " + dest.pathname + " on " + parse.hostname + " ...")
				c.rename(parse.pathname, dest.pathname, function renamedFileOnSftp(err) {
					callback(err);
				});
			}
			else {
				callback(new Error("Failed to move: " + oldPath + "\nNo connection open to SFTP on " + parse.hostname + " !"));
			}
		}
		else {
			// It's a local folder
			var fs = require("fs");
			fs.rename(oldPath, newPath, function(err) {
				if(err) {
					if(err.code == "EISDIR") {
						err = new Error("Make sure " + newPath + " is not already a directory! " + err.message);
						err.code = "EISDIR";
					}
				}
				callback(err, {oldPath: oldPath, newPath: newPath});
			});
		}
	}
	else {
		// Assume it's a file'
		API.readFromDisk(user, {path: oldPath, returnBuffer: true}, function fileRead(err, read) {
		if(err) return callback(err);
		
		if(!Buffer.isBuffer(read.data)) throw new Error("readFromDisk did not give a Buffer! typeof read.data=" + typeof read.data);
		
		API.saveToDisk(user, {path: newPath, text: read.data, inputBuffer: true, public: json.public}, function fileWrite(err, write) {
			if(err) return callback(err);
			
			API.deleteFile(user, {path: oldPath}, function fileDelete(err) {
				if(err) return callback(err);
				else callback(err, {oldPath: oldPath, newPath: newPath});
			});
		});
	});
	}
}

API.getFileSizeOnDisk = function getFileSizeOnDisk(user, json, callback) {
	
	var path = json.path;
	
	// Check path for protocol
	var url = require("url");
	var parse = url.parse(path);
	
	if(parse.protocol == "ftp:" || parse.protocol == "ftps:") {
		
		if(user.remoteConnections.hasOwnProperty(parse.hostname)) {
			
			var c = user.remoteConnections[parse.hostname].client;
			
			console.log("Getting file size from FTP server: " + parse.protocol + parse.hostname + parse.pathname);
			
			// Asume the FTP server has support for RFC 3659 "size"
			c.size(parse.pathname, function gotFtpFileSize(err, size) {
				if(err) {
					console.warn(err.message);
					callback(err);
				}
				else {
					callback(null, {size: size});
				}
			});
		}
		else {
			// Should we give an ENOENT here ?
			callback(new Error("Failed to get file size for: " + path + "\nNo connection open to FTP on " + parse.hostname + " !"));
		}
	}
	else if(parse.protocol == "sftp:") {
		
		if(user.remoteConnections.hasOwnProperty(parse.hostname)) {
			
			var c = user.remoteConnections[parse.hostname].client;
			
			console.log("Getting file size from SFTP server: " + parse.pathname);
			
			c.stat(parse.pathname, function gotSftpFileSize(err, stat) {
				
				if(err) {
					callback(err);
				}
				else {
					callback(null, {size: stat.size});
				}
			});
		}
		else {
			callback(new Error("Failed to get file size for: " + path + "\nNo connection open to SFTP on " + parse.hostname + " !"));
		}
	}
	else {
		
		// It's a normal file path
		
		var module_path = require("path");
		if(!module_path.isAbsolute(path)) {
			var error = new Error("Not an absolute path: " + path);
			error.code = "NOT_ABSOLUTE";
			return callback(error);
		}
		
		var path = user.translatePath(json.path);
		if(path instanceof Error) return callback(path);
		
		
		var fs = require("fs");
		
		fs.stat(path, checkSize);
		
	}

	function checkSize(err, stats) {
		
		if(err) callback(err);
		else callback(null, {size: stats["size"]});
		
	}
}



API.saveToDisk = function saveToDisk(user, json, saveToDiskCallback) {

	if(json.path == undefined) return saveToDiskCallback(new Error("json.path=" + json.path));
	
	var path = json.path;
	
	var inputBuffer = json.inputBuffer || false;
	var encoding = json.encoding || "utf-8";
	var text = json.text; // string, or (if json.inputBuffer==true) Buffer
	
	if(typeof inputBuffer != "boolean") throw new Error("saveToDisk: Error: inputBuffer (" + (typeof inputBuffer) + ") should be a Boolean (true or false)");
	if(!inputBuffer && !UTIL.isString(text)) throw new Error("saveToDisk: Error: text (" + (typeof text) + ") should be a string when inputBuffer is false");
	if(inputBuffer && !Buffer.isBuffer(text)) throw new Error("saveToDisk: Error: text (" + (typeof text) + ") should be a Buffer when inputBuffer is true");

// Check path for protocol
var url = require("url");
var parse = url.parse(path);
var hostname = parse.hostname;
var protocol = parse.protocol;
var pathname = parse.pathname;

	if(!json.public && pathname && pathname.slice(1,7) == "wwwpub") json.public = true; 
	
console.log("Saving to disk ... protocol: " + protocol + " hostname=" + hostname + " pathname=" + pathname);

	var crypto = require('crypto');
	var shaSum = crypto.createHash('sha256');
	shaSum.update(text);
	var hash = shaSum.digest('hex')
	
if(protocol == "ftp:" || protocol == "ftps:") {
	
	if(ftpBusy) {
		console.log("FTP is busy. Queuing upload of pathname=" + pathname + " ...");
		ftpQueue.push(function() { uploadFTP(pathname, text); });
	}
	else {
		ftpBusy = true;
		console.log("FTP is ready. Uploading pathname=" + pathname + " ...");
		uploadFTP(pathname, text);
	}
}
else if(protocol == "sftp:") {
	
	if(user.remoteConnections.hasOwnProperty(hostname)) {
		
		var c = user.remoteConnections[hostname].client;
		
		var input = inputBuffer ? text : new Buffer(text, encoding);
		var destPath = pathname;
		var options = {encoding: encoding};
		// Could also use sftp.createWriteStream
		console.log("Waiting for SFTP ...");
		c.writeFile(destPath, input, options, function sftpWrite(err) {
			if(err) {
				console.warn("Failed to save to path= " + path + "\n" + err.message);
				saveToDiskCallback(err);
			}
			else {
				console.log("Saved " + destPath + " on SFTP " + hostname);
					saveToDiskCallback(null, {path: path, hash: hash});
				}
				
			});
			
		}
		else {
			saveToDiskCallback(new Error("Failed to save to path=" + path + "\nNo connection to SFTP on " + hostname + ""));
		}
	}
	else {
		
		// Asume local file-system
		
		var module_path = require("path");
		if(!module_path.isAbsolute(path)) {
			var error = new Error("Not an absolute path: " + path);
			error.code = "NOT_ABSOLUTE";
			return saveToDiskCallback(error);
		}
		
		var path = user.translatePath(json.path);
		if(path instanceof Error) return saveToDiskCallback(path);
		
		var fs = require("fs");
		
		var options = {
			encoding: encoding,
			mode: DEFAULT_FILE_MODE
		}
		
		if(json.public) {
			// Make it so everyone can read it
			options.mode = parseInt("0777", 8); // 1 = execute, 2 = write, 4 = read
			
			// note: The file permissions wont change if the file already exists!
		}
		
		console.log("saveToDisk: path=" + path + " options=" + JSON.stringify(options) + " json.public=" + json.public);
		
		fs.writeFile(path, text, options, function(err) {
			//console.log("Attempting saving to local file system: " + path + " ...");
			
			if(err) {
				console.warn("Unable to save " + path + "!");
				
				if(err.code == "EISDIR") saveToDiskCallback(new Error("Make sure " + path + " is not a directory! " + err.message));
				else saveToDiskCallback(err);
			}
			else {
				//console.log("The file was successfully saved: " + path + "");
				saveToDiskCallback(null, {path: user.toVirtualPath(path), hash: hash});
			}
		});
	}
	
	
	function uploadFTP(pathname, text) {
		console.log("Uploading to FTP ... pathname=" + pathname + " inputBuffer=" + inputBuffer);
		
		if(user.remoteConnections.hasOwnProperty(hostname)) {
			
			var ftpClient = user.remoteConnections[hostname].client;
			
			// ftp put wants a buffer. Convert to buffer if it's not a buffer!
			var input = inputBuffer ? text : new Buffer(text, encoding);
			var useCompression = false;
			
			ftpClient.put(input, pathname, useCompression, putFtpDone);
			
		}
		else {
			saveToDiskCallback(new Error("Failed to save to path=" + path + "\nNo connection to FTP on " + hostname + " !"));
			runFtpQueue();
		}
		
		function putFtpDone(err) {
			if(err) {
				console.warn("Failed to save pathname= " + pathname + "\n" + err);
				saveToDiskCallback(err);
				runFtpQueue();
			}
			else {
				
				console.log("Successfully saved pathname=" + pathname + "");
				
				saveToDiskCallback(null, {path: path, hash: hash});
				
				runFtpQueue();
			}
			
		}
		
	}
}



API.listFiles = function listFiles(user, json, listFilesCallback) {
	if(listFilesCallback == undefined) throw new Error("Need to specify a callback!");

	var pathToFolder = json.pathToFolder;
	
	if(!pathToFolder) return listFilesCallback(new Error("No pathToFolder defined!"));

	pathToFolder = UTIL.trailingSlash(pathToFolder);
	
	pathToFolder = user.translatePath(pathToFolder);
	if(pathToFolder instanceof Error) return listFilesCallback(pathToFolder);
	
	
	/*
		Try to get the file list in the same format regardless of protocol!
		
		type - string - A single character denoting the entry type: 'd' for directory, '-' for file (or 'l' for symlink on *NIX only).
		name - string - File or folder name
		path - string - Full path to file/folder
		size - float - The size of the entry in bytes.
		date - Date - The last modified date of the entry.
		
		
	*/
	
	var url = require('url');
	var parse = url.parse(pathToFolder);
	var protocol = parse.protocol;
	var hostname = parse.hostname;
	var pathname = parse.pathname;
	
	//console.log("listFiles: protocol=" + protocol + " pathToFolder=" + pathToFolder);
	
	if(protocol == "ftp:" || protocol == "ftps:") {
		// ### List files using FTP protocol
		
		if(ftpBusy) {
			console.log("FTP is busy. Queuing file-list of pathname=" + pathname + " ...");
			ftpQueue.push(function() { listFilesFTP(pathname); });
		}
		else {
			ftpBusy = true;
			console.log("FTP is ready. Listing files in pathname=" + pathname + " ...");
			listFilesFTP(pathname);
		}
	}
	else if(protocol == "sftp:") {
		// ### List file using SFTP protocol
		if(user.remoteConnections.hasOwnProperty(hostname)) {
			
			var c = user.remoteConnections[hostname].client;
			
			console.log("Initiating folder read on SFTP " + hostname + ":" + pathname);
			
			// SFTP can list files in any folder. So we do not have to make sure the path is the same as the working directory (like with ftp)
			// hmm, it seems we can only do readdir once on each folder
			var b = c.readdir(pathname, function sftpReadDir(err, folderItems) {
				
				//UTIL.getStack("XXX");
				
				console.log("Reading folder: " + pathname + " ...");
				
				if(err) {
					listFilesCallback(err);
					listFilesCallback = null;
				}
				else {
					
					//console.log(JSON.stringify(folderItems, null, 2));
					
					var list = [];
					var path = "";
					var type = "";
					
					for(var i=0; i<folderItems.length; i++) {
						path = pathToFolder + folderItems[i].filename; // Asume pathToFolder has a trailing slash
						type = folderItems[i].longname.substr(0, 1);
						
						if(type == "d") path = UTIL.trailingSlash(path);
						
						//console.log("path=" + path);
						list.push({
						type: type, 
						name: folderItems[i].filename, 
						path: path, 
						size: parseFloat(folderItems[i].attrs.size), 
						date: new Date(folderItems[i].attrs.mtime*1000)
						});
					}
					
					listFilesCallback(null, list);
					listFilesCallback = null;
					
				}
				
			});
			
			//console.log("b=" + b);
			
		}
		else {
			listFilesCallback(new Error("Unable to read " + pathname + " on " + hostname + "\nNot connected to SFTP on " + hostname + " !"));
			listFilesCallback = null;
		}
	}
	else {
		// ### List files using "normal" file system
		var fs = require("fs");
		//console.log("Reading directory=" + pathToFolder);
		fs.readdir(pathToFolder, function readdir(err, folderItems) {
			if(err) {
				listFilesCallback(err);
				listFilesCallback = null;
			}
			else {
				var filePath;
				var list = [];
				var statCounter = 0;
				if(folderItems.length == 0) {
					// It's an emty folder
					listFilesCallback(null, list);
					listFilesCallback = null;
				}
				else {
					var path = require("path");
					
					for(var i=0; i<folderItems.length; i++) {
						
						// Check item name for encoding problems �
						
						stat(folderItems[i], path.join(pathToFolder, folderItems[i]));
						// We do not know if it's a folder or file yet, folderItems is just an array of strings, we have to wait for stat
					}
				}
			}
			
			function stat(fileName, filePath) {
				//console.log("Making stat: " + filePath + "");
				
				fs.stat(filePath, function stat(err, stats) {
					
					var type = "";
					var size;
					var mtime;
					var problem = "";
					
					if(stats) {
						size = stats.size;
						mtime = stats.mtime;
						
						if(stats.isFile()) {
							type = "-";
						}
						else if(stats.isDirectory()) {
							type = "d";
							filePath = UTIL.trailingSlash(filePath);
						}
					}
					
					if(err) {
						
						var knownErrors = [
							"EPERM", // operation not permitted
							"EBUSY", // resource busy or locked
							"ENOENT", // no such file or directory
							"ENOTCONN", // socket is not connected, stat '/googleDrive'
							"UNKNOWN", // Windows 10 error (probably access denied)
							"ELOOP" // too many symbolic links encountered
						];
						
						if(knownErrors.indexOf(err.code) != -1) {
							problem = err.code;
							type = "*"
						}
						else {
							if(listFilesCallback) listFilesCallback(err);
							listFilesCallback = null;
							return;
						}
					}
					
					//console.log("stat: " + stats);
					
					var systemFolders = [
						"/dev/", 
						"/usr/", 
						"/lib/", 
						"/lib64/", 
						"/bin/", 
						"/sbin/",
						"/sys/",
						"/etc/", 
						"/proc/", 
						"/run/", 
						//"/sock/", 
						"/.webide/", 
						"/.config/", 
						"/.npm/", 
						"/.ssh/",
						"/.bash_history",
						"/.node_repl_history",
						"/.npm-packages/"
					]; // Ignore mounted files and folders
					if(!user.chrooted || systemFolders.indexOf(filePath) == -1) {
						list.push({type: type, name: fileName, path: user.toVirtualPath(filePath), size: size, date: mtime, problem: problem});
					}
					
					statCounter++;
					
					//console.log("Finished stat: " + filePath + " statCounter=" + statCounter + " folderItems.length=" + folderItems.length);
					
					if(statCounter==folderItems.length) {
listFilesCallback(null, list);
						listFilesCallback = null;
					}
					
				});
			}
			
			
		});
		
		
	}
	
	
	function listFilesFTP(pathname) {
		
		if(user.remoteConnections.hasOwnProperty(hostname)) {
			
			var ftpClient = user.remoteConnections[hostname].client;
			
			ftpListFiles(ftpClient);
			
		}
		else {
			listFilesCallback(new Error("Unable to read " + pathname + " on " + hostname + "\nNot connected to FTP on " + hostname + " !"));
			runFtpQueue();
		}
		
		function ftpListFiles(ftpClient) {
			
			console.log("Listing files in '" + parse.pathname + "' on " + parse.protocol + parse.hostname);
			
			ftpClient.list(parse.pathname, function readdirFtp(err, folderItems) {
				if (err) {
					console.warn(err.message);
					listFilesCallback(err);
					runFtpQueue();
				}
				else {
					
					var list = [];
					var path = "";
					var type = "";
					
					//console.log("folderItems=" + JSON.stringify(folderItems, null, 2));
					
					for(var i=0; i<folderItems.length; i++) {
						
						//console.log("name=" + folderItems[i].name);
						
						path = pathToFolder + folderItems[i].name;
						type = folderItems[i].type;
						
						if(type == "d") path = UTIL.trailingSlash(path);
						
						// todo: parse date ?
						list.push({type: type, name: folderItems[i].name, path: path, size: parseFloat(folderItems[i].size), date: folderItems[i].date});
					}
					
					listFilesCallback(null, list);
					
					runFtpQueue();
					
				}
			});
		}
	}
}

// todo: Some sort of suecurity where a working "jail" can be issued to each user
API.workingDirectory = function workingDirectory(user, json, callback) {
	callback(null, {path: user.workingDirectory});
}



API.createPath = function createPath(user, json, createPathCallback) {
	/*
		Traverse the path and try to creates the directories, then check if the full path exists
		
	*/
	
	var lastCharOfPath = json.pathToCreate.substr(json.pathToCreate.length-1);
	
	if(lastCharOfPath != "/" && lastCharOfPath != "\\") return createPathCallback("Last character is not a file path delimiter: " + json.pathToCreate);
	
	var pathToCreate = user.translatePath(json.pathToCreate);
	if(pathToCreate instanceof Error) return createPathCallback(pathToCreate);
	
	console.log("json.pathToCreate=" + json.pathToCreate + " pathToCreate=" + pathToCreate);
	
	pathToCreate = UTIL.trailingSlash(pathToCreate);
	
	var url = require('url');
	var parse = url.parse(pathToCreate);
	var protocol = parse.protocol;
	var delimiter = UTIL.getPathDelimiter(pathToCreate);
	var lastChar = pathToCreate.substring(pathToCreate.length-1);
	var hostname = parse.hostname;
	var create = UTIL.getFolders(pathToCreate);
	var errors = [];
	var fullPath = create[create.length-1];
	
	if(protocol) protocol = protocol.replace(/:/g, "").toLowerCase();
	
	console.log("create=" + JSON.stringify(create) + " hostname=" + hostname + " pathToCreate=" + pathToCreate + " parse=" + JSON.stringify(parse));
	
	create.shift(); // Don't bother with the root
	
	// Execute mkdir in order !
	if(create.length == 0) {
		console.warn("No path to create! fullPath=" + fullPath);
		createPathCallback(null, {path: fullPath});
	}
	else executeMkdir(create.shift());
	
	function executeMkdir(folder) {
		// This is a recursive function!
		createPathSomewhere(folder, json.public, function(err, path) {
			if(err) {
errors.push(err.message + " path=" + path);
				//console.warn("Failed to create path=" + path + "\n" + err.message);
			}
			else {
				//console.log("Successfully created path=" + path);
			}
			
			if(create.length > 0) executeMkdir(create.shift());
			else done();
			
			});
	}
	
	function done() {
		// Check if the full path exists
		
		pathToCreate = user.toVirtualPath(pathToCreate); // API.listFiles wants a virtual path!
		
		API.listFiles(user, {pathToFolder: pathToCreate}, listFileResult);
		
		function listFileResult(err, files) {
			
			if(err) {
				console.warn("List failed! " + err.message + " pathToCreate=" + pathToCreate);
				var errorMsg = "Failed to create path=" + pathToCreate + "\n" + err.message;
				for(var i=0; i<errors.length; i++) {
					errorMsg += "\n" + errors[i];
				}
				
				createPathCallback(new Error(errorMsg));
			}
			else {
				createPathCallback(null, {path: fullPath})
			}
		}
	}
	
	function createPathSomewhere(path, publicFolder, createPathSomewhereCallback) {
		
		// ## mkdir ...
		
		console.log("Creating path=" + path);
		
		if(path.indexOf("//") > 6) {
			path = path.replace(/\/\/+/g, "/"); // Remove double slashes
			
			if(protocol) {
				// Re-add the slash after the protocol
				path = path.replace(protocol + ":/", protocol + "://");
			}
			
			console.warn("Sanitized path=" + path + " pathToCreate=" + pathToCreate);
		}
		
		if(protocol) {
			// We only want the path!
			path = url.parse(path).pathname;
		}
		
		if(protocol == "ftp" || protocol == "ftps") {
			// ### Create a directory using FTP protocol
			
			if(ftpBusy) {
				console.log("FTP is busy. Queuing mkdir of path=" + path + " ...");
				ftpQueue.push(function() { createPathFTP(path); });
			}
			else {
				ftpBusy = true;
				console.log("FTP is ready. Creating path=" + path + " ...");
				createPathFTP(path);
			}
			
			
		}
		else if(parse.protocol == "sftp:") {
			// ### Create a directory using SFTP protocol
			if(user.remoteConnections.hasOwnProperty(parse.hostname)) {
				
				var c = user.remoteConnections[parse.hostname].client;
				
				var b = c.mkdir(path, function (err, folderItems) {
					
					//UTIL.getStack("XXX");
					
					if(err) createPathSomewhereCallback(err, path);
					else createPathSomewhereCallback(null, path);
					
					
				});
				
				// b = false : If you should wait for the continue event before sending any more traffic.
				
				//console.log("b=" + b);
				
			}
			else {
				createPathCallback(new Error("Unable to create " + path + " on " + hostname + "\nNot connected to SFTP on " + hostname + " !"));
			}
		}
		else {
			// ### Create a directory using "normal" file system
			var fs = require("fs");
			
			
			var folderMode = DEFAULT_FOLDER_MODE;
			
			if(publicFolder) {
				// Make it so everyone can read it
				folderMode = parseInt("0777", 8);
				
				// note: The file permissions wont change if the file already exists!
				}
			
			
			fs.mkdir(path, folderMode, function(err) {
				
				if(err) createPathSomewhereCallback(err, path);
				else createPathSomewhereCallback(null, path);
			});
		}
		
		
		function createPathFTP(path) {
			
			console.log("Creating FTP path=" + path)
			
			if(user.remoteConnections.hasOwnProperty(hostname)) {
				
				var c = user.remoteConnections[hostname].client;
				
				// ftp mkdir
				c.mkdir(path, function(err) {
					
					console.log("Done creating FTP path=" + path);
					
					if(err) createPathSomewhereCallback(err, path);
					else createPathSomewhereCallback(null, path);
					
					runFtpQueue();
					
				});
				
			}
			else {
				createPathCallback(new Error("Unable to create path=" + path + " on " + hostname + "\nNot connected to FTP on " + hostname + " !"));
				runFtpQueue();
			}
		}
		
	}
}


API.connect = function connect(user, json, callback) {
	
	var protocol = json.protocol;
	var serverAddress = json.serverAddress;
	var username = json.user;
	var passw = json.passw;
	var keyPath = json.keyPath;
	var workingDir = json.workingDir;
	
	if(protocol == undefined) return callback("No protocol defined! protocol=" + protocol);
	if(serverAddress == undefined) return callback("No serverAddress defined! serverAddress=" + serverAddress);
	if(username == undefined) return callback("No user defined! username=" + username);
	if(workingDir !== undefined) return callback("workingDir parameter not yet implemented! workingDir=" + workingDir);
	
	if(protocol.indexOf(":") != -1) {
		console.warn("Removing : (colon) from protocol=" + protocol);
		protocol = protocol.replace(/:/g, "");
	}
	
	protocol = protocol.toLowerCase();
	
	console.log("protocol=" + protocol);
	
	var supportedRemoteProtocols = ["ftp", "ftps", "sftp"];
	
	if(supportedRemoteProtocols.indexOf(protocol) == -1) throw new Error("Protocol=" + protocol + " not supported! supportedRemoteProtocols=" + JSON.stringify(supportedRemoteProtocols)); 
	
	if(protocol == "ftp" || protocol == "ftps") {
		
		if(ftpQueue.length > 0) {
			console.warn("Removing " + ftpQueue.length + " items from the FTP queue");
			ftpQueue.length = 0;
		}
		
		var Client = module_ftp;
		user.remoteConnections[serverAddress] = {client: new Client(), protocol: protocol};
		var ftpClient = user.remoteConnections[serverAddress].client;
		ftpClient.on('ready', function() {
			console.log("Connected to FTP server on " + serverAddress + " !");
			ftpClient.pwd(function(err, dir) {
				if(err) throw err;
				user.changeWorkingDir(protocol + "://" + serverAddress + dir.replace("\\", "/"));
				
				// Create disconnect function
				user.remoteConnections[serverAddress].close = function disconnectFTP() {
					ftpClient.end();
					delete user.remoteConnections[serverAddress];
					
					console.log("Dissconnected from FTP on " + serverAddress + "");
				};
				
				callback(null, {workingDirectory: user.workingDirectory});
				callback = null; // Don't callback again when the connection timeouts
				
			});
			
		});
		
		ftpClient.on('error', function(err) {
			
			if(callback) callback(err);
			else user.send(err.message + " (serverAddress=" + serverAddress + ")");
			
			callback = null;
			user.remoteConnectionClosed("ftp", serverAddress);
			
		});
		
		ftpClient.on('close', function(hadErr) {
			user.send("Connection to FTP on " + serverAddress + " closed.", "REMOTE_CONNECTION_CLOSE");
			
			user.remoteConnectionClosed("ftp", serverAddress);
			
		});
		
		var options = {host: serverAddress, user: username, password: passw};
		
		if(protocol == "ftps") {
			options.secure = true;
			
			// Some times the cert is lost!? So we need to override checkServerIdentity to return undefined instead of throwing an error: Cannot read property 'CN' of undefined
			// https://nodejs.org/api/tls.html#tls_tls_connect_options_callback
			
			options.secureOptions = {
				checkServerIdentity: function(servername, cert) {
					console.log("Checking server identity for servername=" + servername);
					
					if(Object.keys(cert).length == 0) console.warn("No cert attached!");
					else {
						// Do some checking?
						//console.log(JSON.stringify(cert));
					}
					
					return undefined;
					
				}
			}
		}
		console.log("Connecting to " + options.host + " ...");
		ftpClient.connect(options);
	}
	
	// note: SSH (shell) not yet supported. Use SFTP instead!
	else if(protocol == "ssh") {
		
		sshConnect(function sshConnected(err, sshClient, workingDir) {
			if(err) return callback(err);
				
				user.remoteConnections[serverAddress] = {client: sshClient, protocol: protocol};
				
				// Create disconnect function
				user.remoteConnections[serverAddress].close = function disconnectSSH() {
					sshClient.end();
					delete user.remoteConnections[serverAddress];
					
					console.log("Dissconnected from SSH on " + serverAddress + "");
				};
				
				user.changeWorkingDir(workingDir);
				
				callback(null, {workingDirectory: user.workingDirectory});
				callback = null; // Don't callback again when the connection timeouts
			
		});
		
	}
	else if(protocol == "sftp") {
		
		sshConnect(function sshConnected(err, sshClient, workingDir) {
			if(err) return callback(err);
			
			// Initiate "SFTP mode"
				sshClient.sftp(function(err, sftpClient) {
					if (err) {
						sshClient.end();
						return callback(err);
}
					else {
						user.remoteConnections[serverAddress] = {client: sftpClient, protocol: protocol};
						user.changeWorkingDir(workingDir);
						
						console.log("Connected to SFTP on " + serverAddress + " . Working directory is: " + user.workingDirectory);
						
						// Create disconnect function
						user.remoteConnections[serverAddress].close = function disconnectSFTP() {
							sshClient.end();
							delete user.remoteConnections[serverAddress];
							
							console.log("Dissconnected from SFTP on " + serverAddress + "");
						};
						
						callback(null, {workingDirectory: user.workingDirectory});
						callback = null; // Don't callback again when the connection timeouts
					}
				});
			
		});
	}
	else {
		throw new Error("Protocol not supported: " + protocol);
	}
	
	
	function sshConnect(cb) {
		// Connects to a SSH server and sets the working directory, returns the "connection" in the cb callback
		
		var auth = {
			host: serverAddress,
			port: 22,
			username: username,
		}
		
		if(keyPath) {
			// Connect using key
			API.readFromDisk(user, {path: keyPath}, function readKey(err, json) { // Read key
				if(err) return cb(err);
				
				var path = json.path
				var keyStr = json.data;
				
				auth.passphrase = passw;
				auth.privateKey = keyStr;
				try {
					connect();
				}
				catch(err) {
					cb(err);
				}
			});
		}
		else {
			// Connect using password
			auth.password = passw;
			connect();
		}
		
		function connect() {
			var Client = module_ssh2.Client;
			
			var c = new Client();
			c.on('ready', function() {
				console.log('Client :: ready');
				
				c.exec('pwd', function(err, stream) {
					if (err) throw err;
					var dir = "";
					stream.on('close', function(code, signal) {
						//console.log('Stream :: close :: code: ' + code + ', signal: ' + signal);
						
						console.log("SFTP pwd result: dir=" + dir);
						
						// Problem: "This service allows sftp connections only"
						var workingDir = UTIL.trailingSlash(protocol + "://" + serverAddress);
						if(dir.charAt(0) == "/") {
						
						// Chop off the newline character
						dir = dir.substring(0, dir.length-1);
						
							var workingDir = workingDir + dir.replace("\\", "/");
						}
						
						cb(null, c, workingDir);
						cb = null;

						//c.end();
					}).on('data', function(data) {
						console.log('SFTP pwd stdout: ' + data);
						dir += data;
					}).stderr.on('data', function(data) {
//cb(new Error("Error executing pwd on SSH:" +  serverAddress + "\n" + data));
						user.send("Error executing pwd on SSH:" +  serverAddress + "\n" + data);
						console.warn('SFTP pwd stderr: ' + data);
					});
				});
				
			}).on('error', function(err) {
				cb(err);
cb = null;
				
				if(err.message == "All configured authentication methods failed") {
					user.send("Problem connecting to SSH on " + serverAddress + "\n" + err.message + "\nYou might need a key!");
				}
				else {
					user.send("Problem connecting to SSH on " + serverAddress + "\n" + err.message);
				}
				user.remoteConnectionClosed("ssh", serverAddress);
				
			}).on('end', function(msg) {
				user.send("Disconnected from SSH on " + serverAddress + "\nMessage: " + msg, "REMOTE_CONNECTION_CLOSE");
				
				user.remoteConnectionClosed("ssh", serverAddress);
				
			}).connect(auth);
		}
	}
}

API.disconnect = function disconnect(user, json, callback) {
	// Disconnect remove connection
	
	var protocol = json.protocol;
	var serverAddress = json.serverAddress;
	
	if(!user.remoteConnections.hasOwnProperty(serverAddress)) return callback(new Error("Unknown connection: serverAddress=" + serverAddress));
	
	user.changeWorkingDir(user.defaultWorkingDirectory);
	
	user.remoteConnections[serverAddress].close();
	
	callback(null, {workingDirectory: user.workingDirectory});
	
}

API.setWorkingDirectory = function setWorkingDirectory(user, json, callback) {
	
	/*
		Working directory can be on a remote file-system!
	  
	*/
	
	
	var path = user.translatePath(json.path);
	if(path instanceof Error) return callback(path);
	
	callback(null, {workingDirectory:path});
	
	/*
	var fs = require("fs");
	fs.stat(path, function (err, stats){
		if (err) {
			console.log("Error when running stat on path=" + path);
			return callback(err);
		}
		if (!stats.isDirectory()) callback(new Error('Not a directory: path=' + path));
		else {
			path = user.changeWorkingDir(path);
			
			callback(null, {workingDirectory:path});
		}
	});
	*/
}




API.storageGetAll = function storageGetAll(user, json, callback) {
	
	if(user.storage) {
		callback(null, {storage: user.storage});		
	}
	else {
		
		user.loadStorage(function(err, data) {
			if(err) callback(err);
			else callback(null, {storage: data});
		});

	}

}


API.storageSet = function storageSet(user, json, callback) {
	
	var itemName = json.item;
	var value = json.value;
	
	if(itemName == undefined) return callback(new Error("item=" + itemName + " can not be null or undefined!"));
	if(value === undefined) return callback(new Error("value must be defined!"));
	
	if(!user.storage) {
		user.loadStorage(function(err, data) {
			if(err) callback(err);
			else save();
		});
	}
	else save();
	
	function save() {
		user.storage[itemName] = value;
		user.saveStorageItem(itemName, function(err) {
			if(err) callback(err);
			else callback(null, {saved: itemName});
		});
	}
	
}

API.storageRemove = function storageRemove(user, json, callback) {
	
if(!user.storage) return callback("User storage not yet loaded!");

	var itemName = json.item;
	
	if(itemName == undefined) return callback(new Error("item=" + itemName + " can not be null or undefined!"));
	
	if(!user.storage.hasOwnProperty(itemName)) {
		var error = new Error("Item=" + itemName + " is already gone from the storage!");
		error.code = "ENOENT";
		return callback(error);
	}
	
	delete user.storage[itemName];
	
	user.removeStorageItem(itemName, function(err) {
		if(err) callback(err);
		else callback(null, {removed: itemName});
	});
	
}

API.deleteFile = function deleteFile(user, json, callback) {
	
	var filePath = user.translatePath(json.filePath || json.path);
	if(filePath instanceof Error) return callback(filePath);
	
	// Check path for protocol
	var url = require("url");
	var parse = url.parse(filePath);
	
	if(parse.protocol == "ftp:" || parse.protocol == "ftps:") {
		
		if(user.remoteConnections.hasOwnProperty(parse.hostname)) {
			
			var c = user.remoteConnections[parse.hostname].client;
			
			console.log("Deleting file from FTP server: " + parse.protocol + parse.hostname + parse.pathname);
			
			c.delete(parse.pathname, function ftpFileDeleted(err) {
				if(err) {
					console.warn(err.message);
					callback(err);
				}
				else {
					callback(null, {filePath: json.filePath});
				}
			});
		}
		else {
			// Should we give an ENOENT here ?
			callback(new Error("Failed to delete file: " + filePath + "\nNo connection open to FTP on " + parse.hostname + " !"));
		}
	}
	else if(parse.protocol == "sftp:") {
		
		if(user.remoteConnections.hasOwnProperty(parse.hostname)) {
			
			var c = user.remoteConnections[parse.hostname].client;
			
			console.log("Deleting file from SFTP server: " + parse.pathname);
			
			/*
			for (var m in c) {
				console.log(m + ": " + typeof c[m]);
			}
			*/
			
			c.unlink(parse.pathname, function sftpFileDeleted(err) {
				
				if(err) callback(err);
				else callback(null, {filePath: json.filePath});
				
			});
		}
		else {
			callback(new Error("Failed to delete file: " + filePath + "\nNo connection open to SFTP on " + parse.hostname + " !"));
		}
	}
	else {
		
		// It's a normal file path
		
		var fs = require("fs");
		
		fs.unlink(filePath, function localFileDeleted(err) {
			if(err) callback(err);
			else callback(null, {filePath: json.filePath});
		});
		
	}
	}


API.deleteDirectory = function deleteDirectory(user, json, callback) {
	
	var directory = json.directory;
	var recursive = json.recursive || false;
	
	// Check path for protocol
	var url = require("url");
	var parse = url.parse(directory);
	
	if(parse.protocol == "ftp:" || parse.protocol == "ftps:") {
		
		if(user.remoteConnections.hasOwnProperty(parse.hostname)) {
			
			var c = user.remoteConnections[parse.hostname].client;
			
			console.log("Deleting directory from FTP server: " + parse.protocol + parse.hostname + parse.pathname);
			
			c.rmdir(parse.pathname, recursive, function ftpDirDeleted(err) {
				if(err) {
					console.warn(err.message);
					callback(err);
				}
				else {
					callback(null, {directory: json.directory});
				}
			});
		}
		else {
			// Should we give an ENOENT here ?
			callback(new Error("Failed to delete directory: " + directory + "\nNo connection open to FTP on " + parse.hostname + " !"));
		}
	}
	else if(parse.protocol == "sftp:") {
		
		if(user.remoteConnections.hasOwnProperty(parse.hostname)) {
			
			var c = user.remoteConnections[parse.hostname].client;
			
			console.log("Deleting directory from SFTP server: " + parse.pathname);
			
			if(recursive) {
				recursiveDeleteSftpDir(c, parse.pathname, function dirDeletedRecursive(err) {
					if(err) callback(err);
					else callback(null, {directory: json.directory});
				});
			}
			else {
				
			c.rmdir(parse.pathname, function sftpDirDeleted(err) {
				
				if(err) callback(err);
				else callback(null, {directory: json.directory});
				
			});
			}
			
		}
		else {
			callback(new Error("Failed to delete directory: " + directory + "\nNo connection open to SFTP on " + parse.hostname + " !"));
		}
	}
	else {
		
		// It's a normal file path
		
		var directory = user.translatePath(json.directory);
		if(directory instanceof Error) return callback(directory);
		
		var fs = require("fs");
		
		if(recursive) {
			recursiveDeleteLocalDir(directory, function dirDeletedRecursive(err) {
				if(err) callback(err);
				else callback(null, {directory: json.directory});
			});
		}
		else {
		fs.rmdir(directory, function localFileDeleted(err) {
			if(err) callback(err);
			else callback(null, {directory: json.directory});
		});
		}
	}
	
	function recursiveDeleteSftpDir(sftpClient, pathToFolder, recursiveDeleteSftpDirCallback) {
		
			if(pathToFolder.substr(pathToFolder.length-1) != "/" && pathToFolder.substr(pathToFolder.length-1) != "\\") {
			return recursiveDeleteSftpDirCallback("pathToFolder=" + pathToFolder + " does not end with a trailing slash!");
			}
			
			var gotError = false;
		var filesToBeDeleted = 0;
		var foldersToBeDeleted = 0;
		
			console.log("SFTP recursive delete: pathToFolder=" + pathToFolder);
			var b = sftpClient.readdir(pathToFolder, function sftpReadDir(err, folderItems) {
				
				if(gotError) return; // If we have already got an error while deleting the content of this directory
				
				if(!err) {
				//console.log(JSON.stringify(folderItems, null, 2));
					for(var i=0, path = "", type = ""; i<folderItems.length; i++) {
					path = pathToFolder + folderItems[i].filename; // Asume pathToFolder has a trailing slash
					type = folderItems[i].longname.substr(0, 1);
					check(type, path);
					}
					}
				
				allFilesAndFoldersDeletedMaybe(err);
			
		});
		
		function check(type, path) {
			if(type == "d") {
				foldersToBeDeleted++;
				path = UTIL.trailingSlash(path);
				recursiveDeleteDir(path, function (err) {
					foldersToBeDeleted--;
					allFilesAndFoldersDeletedMaybe(err);
				});
			}
			else {
				filesToBeDeleted++;
				sftpClient.unlink(path, function sftpFileDeleted(err) {
					filesToBeDeleted--;
					allFilesAndFoldersDeletedMaybe(err);
				});
			}
		}
		
			function allFilesAndFoldersDeletedMaybe(err) {
				if(gotError) return; // If we have got an error, it means we have already called the callback
				
				if(err) {
					// Got an error when deleting a file or folder in the directory
					gotError = err;
				recursiveDeleteSftpDirCallback(err);
				}
				else {
				// Make sure the directory is emty before deleting it 
				if(filesToBeDeleted === 0 && foldersToBeDeleted === 0) {
					sftpClient.rmdir(pathToFolder, function sftpDirDeleted(err) {
						recursiveDeleteSftpDirCallback(err);
						});
				}
			}
			}
	}
	
	function recursiveDeleteLocalDir(pathToFolder, recursiveDeleteLocalDirCallback) {
		var fs = require("fs");
		
		if(pathToFolder.substr(pathToFolder.length-1) != "/" && pathToFolder.substr(pathToFolder.length-1) != "\\") {
			return recursiveDeleteLocalDirCallback("pathToFolder=" + pathToFolder + " does not end with a trailing slash!");
		}
		
		var gotError = false;
		var filesToBeDeleted = 0;
		var foldersToBeDeleted = 0;
		var pathsToStat = 0;
		
		console.log("Local file-system recursive delete: pathToFolder=" + pathToFolder);
		
		fs.readdir(pathToFolder, function readdir(err, folderItems) {
			
			if(err) allFilesAndFoldersDeletedMaybe(err);
			else {
				for(var i=0, path=""; i<folderItems.length; i++) {
				// We do not know if it's a folder or file yet, folderItems is just an array of strings, we have to wait for stat
					path = pathToFolder + folderItems[i];
					stat(path);
					}
				allFilesAndFoldersDeletedMaybe(null);
			}
			
		});
		
		function stat(path) {
			pathsToStat++;
			fs.stat(path, function (err, stats) {
				pathsToStat--;
				
				if(gotError) return; // If we have already got an error while deleting the content of this directory
				
				if(stats) {
					
					if(stats.isFile()) {
						//console.log("It's a file! path=" + path);
						filesToBeDeleted++;
						fs.unlink(path, function localFileDeleted(err) {
							filesToBeDeleted--;
							allFilesAndFoldersDeletedMaybe(err);
						});
					}
					else if(stats.isDirectory()) {
						//console.log("It's a folder! path=" + path);
						foldersToBeDeleted++;
						path = UTIL.trailingSlash(path);
						recursiveDeleteLocalDir(path, function (err) {
							foldersToBeDeleted--;
							allFilesAndFoldersDeletedMaybe(err);
						});
					}
				}
				
				allFilesAndFoldersDeletedMaybe(err);
				
			});
		}
		
	
		function allFilesAndFoldersDeletedMaybe(err) {
			
			//console.log("allFilesAndFoldersDeletedMaybe? gotError=" + gotError + " err=" + err + " pathsToStat=" + pathsToStat + " filesToBeDeleted=" + filesToBeDeleted + " foldersToBeDeleted=" + foldersToBeDeleted);
			
			if(gotError) return; // If we have got an error, it means we have already called the callback
			
			if(err) {
				// Got an error when deleting a file or folder in the directory
				gotError = err;
				recursiveDeleteLocalDirCallback(err);
			}
			else {
				// Make sure the directory is emty before deleting it
				if(filesToBeDeleted === 0 && foldersToBeDeleted === 0 && pathsToStat === 0) {
					fs.rmdir(pathToFolder, function localFileDeleted(err) {
					recursiveDeleteLocalDirCallback(err);
					});
					}
			}
		}
	}
	
}



API.findReplaceInFiles = function findReplaceInFiles(user, json, findReplaceInFilesCallback) {
	
	/*
		Finds or replaces inside files
		Can be run on remote file systems (ftp,sftp,ftps)
		
		Streams vs non streams: Streams use less memory but is harder to search with multi-line regexp
		Hopefully each file will not be that big and we'll be able to load each file into memory.
		
		Performance optimization:
		maxFilesToSearchAtTheSameTime = 5; Found 3 match(es) in 2655/10365 file(s) searched in 186s.
		maxFilesToSearchAtTheSameTime = 50; Found 3 match(es) in 2655/10365 file(s) searched in 41.66s.
		
	*/
	
	var searchPath = json.searchPath;
	if(searchPath == undefined) return findReplaceInFilesCallback(new Error("searchPath=" + searchPath + " is not defined!"));
	
	var searchString = json.searchString;
	if(searchString == undefined) return findReplaceInFilesCallback(new Error("searchString=" + searchString + " is not defined!"));
	if(searchString == "") return findReplaceInFilesCallback(new Error("searchString=" + searchString + " can not be empty!"));
	
	try {
		var testSearchString = new RegExp(fileFilter);
	}
	catch(err) {
		return findReplaceInFilesCallback(new Error("Bad RegExp: searchString=" + searchString + ": " + err.message));
	}
	
	var fileFilter = json.fileFilter;
	if(fileFilter == undefined) return findReplaceInFilesCallback(new Error("fileFilter=" + fileFilter + " is not defined!"));
	
	try {
	var fileFilterRegExp = new RegExp(fileFilter);
	}
	catch(err) {
		return findReplaceInFilesCallback(new Error("Bad RegExp: fileFilter=" + fileFilter + ": " + err.message));
	}
	
	var searchSubfolders = json.searchSubfolders || false;
	var maxFolderDepth = json.maxFolderDepth || 20;
	var searchMaxFiles = json.searchMaxFiles || 100000;
	var maxTotalMatches = json.maxTotalMatches || 500;
	var caseSensitive = json.caseSensitive || false;
	var searchSessionId = json.id || 0;
	var showSurroundingLines = json.showSurroundingLines || 2;
	var replaceWith = json.replaceWith;
	
	var totalFiles = 0;
	var filesSearched = 0;
	
	var fileQueue = []; // Files to be searched
	var foldersToRead = 0;
	var totalMatches = 0;
	var totalFilesFound = 0;
	var matches = [];
	var flags = "g"; // Always make a global search!
	var filesBeingSearched = 0;
	var abort = false;
	var done = false;
	var searchSymLinks = true;
	var maxFilesToSearchAtTheSameTime = 20; // Hard drivers are really bad at multi tasking
	var totalFoldersSearched = 0;
	var totalFoldersToSearch = 0;
	var progressInterval = 350;
	var lastProgress = new Date();
	var searchBegin = new Date();
	var totalFilesSearched = 0;
	
	if(!caseSensitive) flags += "i";
	
	FIND_IN_FILES_ABORTED = false;
	
	searchDir(searchPath, 0);
	
	function searchDir(folderPath, folderDepth) {
		
		console.log("Searching: " + folderPath);
		
		if(folderDepth > maxFolderDepth) return console.log("Max folder depth reached! maxFolderDepth=" + maxFolderDepth + " folderDepth=" + folderDepth + " folderPath=" + folderPath);
		
		foldersToRead++;
		folderDepth++;
		totalFoldersToSearch++;
		
		API.listFiles(user, {pathToFolder: folderPath}, function gotFileList(err, fileList) {
			
			if(FIND_IN_FILES_ABORTED) return aborted();
			
			if(err) return abortError(err);
			
			/*
				type - string - A single character denoting the entry type: 'd' for directory, '-' for file (or 'l' for symlink on *NIX only).
				name - string - File or folder name
				path - string - Full path to file/folder
				size - float - The size of the entry in bytes.
				date - Date - The last modified date of the entry.
			*/
			for (var i=0; i<fileList.length; i++) {
				if(fileList[i].type == "d" && searchSubfolders) {
searchDir(fileList[i].path, folderDepth);
				}
				else if(fileList[i].type == "-" || (fileList[i].type == "l" && searchSymLinks)) {
					totalFilesFound++;
					if(fileFilterRegExp.test(fileList[i].path)) fileQueue.push(fileList[i].path);
				}
			}
			
			foldersToRead--;
			totalFoldersSearched++;
			
			doWeHaveAllFiles();
			
		});
		
	}
	
	function doWeHaveAllFiles() {
		if(foldersToRead == 0) {
			// All folders have now been searched!
			totalFiles = fileQueue.length;
			if(fileQueue.length == 0) {
				doneFinish("Found " + totalFilesFound + " files. But none of them math the file filter!");
			}
			else {
				continueSearchFiles();
				}
			}
	}
	
	function continueSearchFiles() {
		
		console.log("continueSearchFiles: fileQueue.length=" + fileQueue.length + " filesBeingSearched=" + filesBeingSearched);
		
		if(FIND_IN_FILES_ABORTED) return aborted();
		if(done) return console.log("Already done! from continueSearchFiles()");
		
		if(totalFiles >= searchMaxFiles) {
			doneFinish("Aborted the search because we reached searchMaxFiles=" + searchMaxFiles + " limit!");
			FIND_IN_FILES_ABORTED = true;
			return;
		}
		else if(totalMatches >= maxTotalMatches) {
			doneFinish("Aborted the search because we reached maxTotalMatches=" + maxTotalMatches + " limit!");
			FIND_IN_FILES_ABORTED = true;
			return;
		}
		else while(fileQueue.length > 0 && filesBeingSearched < maxFilesToSearchAtTheSameTime) searchNextFileInQueue();
		
		doneMaybe();
		
	}
	
	function doneMaybe() {
		
		console.log("doneMaybe: fileQueue.length=" + fileQueue.length + " filesBeingSearched=" + filesBeingSearched);
		
		if(FIND_IN_FILES_ABORTED) return aborted()
		if(done) throw new Error("We should not be calling doneMaybe() if done!");
		
		sendProgress();
		
		if(fileQueue.length == 0 && filesBeingSearched == 0) doneFinish();
		else {
			//continueSearchFiles(); // RangeError: Maximum call stack size exceeded
			setTimeout(continueSearchFiles, 500); // Give a few milliseconds of rest
	}
	}
	
	function searchNextFileInQueue() {
	
		console.log("searchNextFileInQueue: fileQueue.length=" + fileQueue.length + " filesBeingSearched=" + filesBeingSearched);
		
		if(FIND_IN_FILES_ABORTED) return aborted();
		if(done) throw new Error("We should not be calling searchNextFileInQueue() if done==" + done + "!");
		
		var filePath = fileQueue.pop(); // Last in, first out
		
		if(filePath == undefined) {
			if(fileQueue.length == 0) doneFinish();
			else throw new Error("filePath=" + filePath + " fileQueue.length=" + fileQueue.length);
		}
		else searchFile(filePath);
		
	}
	
	function searchFile(filePath) {
		filesBeingSearched++;
		API.readFromDisk(user, {path: filePath}, function readFile(err, json) {
			
			if(err) return abortError(err);
			
			var filePath = json.path;
			var fileContent = json.data;
		
			var myRe = new RegExp(searchString, flags); // Create a new RegExp for each file!
			
			console.log("Searching file: " + filePath);
				
				var result;
				var lastIndex = 0;
			var lastLine = 1;
			var rowsAbove = [];
			var rowsBeneath = [];
			
			while ((result = myRe.exec(fileContent)) !== null) {
					
					totalMatches++;
					
					// Figure out what the line number is
					// Select all text up until the first line break after the search match index
					var firstLineBreakAfterResult = fileContent.indexOf("\n", result.index + result[0].length);
					var textAboveInludingResult = fileContent.slice(  result.lastIndex, firstLineBreakAfterResult  );
					if(textAboveInludingResult.charAt(textAboveInludingResult.length-1) == "\r") textAboveInludingResult = textAboveInludingResult.slice(0, -1);
					var resultRows =  result[0].split(/\r\n|\n/);
					var textAboveInludingResultRows = textAboveInludingResult.split(/\r\n|\n/);
					var lineNr = textAboveInludingResultRows.length - resultRows.length + 1;
					
					lastLine = lineNr;
					
					var lineText = "";
					// Line text can be many lines!
					for (var i=0; i<resultRows.length; i++) {
						lineText = textAboveInludingResultRows.pop() + "\n" + lineText;
					}
					lineText = lineText.trim();
					
					if(matches.indexOf(result[0]) == -1) matches.push(result[0]); // Highlight these later
					
					if(showSurroundingLines) {
						rowsAbove = textAboveInludingResultRows.slice( -showSurroundingLines );
						var index = firstLineBreakAfterResult;
						if(fileContent.charAt(index) == "\r") index++;
						if(fileContent.charAt(index) == "\n") index++;
						rowsBeneath = [];
						for (var i=index; i<fileContent.length; i++) {
							if(fileContent.charAt(i) == "\n") {
								rowsBeneath.push(fileContent.slice(index, i).trim());
								index = i+1;
								if(rowsBeneath.length >= showSurroundingLines) break;
							}
						}
						
					}
					
					//console.log("textAboveInludingResultRows=" + JSON.stringify(textAboveInludingResultRows));
					//console.log("rowsAbove=" + JSON.stringify(rowsAbove));
					
					console.log("Found " + result[0] + " on index=" + result.index + " lastIndex=" + result.lastIndex + " showSurroundingLines=" + showSurroundingLines + " lineNr=" + lineNr + " in file=" + filePath);
					
					var foundInFile = {
						id: searchSessionId,
						text: result[0],
						lineText: lineText,
						index: result.index,
						lineNr: lineNr,
						file: filePath,
						rowsAbove: rowsAbove,
						rowsBeneath: rowsBeneath,
						regExp: myRe.toString()
					};
					
					if(replaceWith) {
					foundInFile.replaceWith = replaceWith;
					// Run replace op on client side, not twice here.
					//foundInFile.replacedWith = result[0].replace(myRe, replaceWith);
				}
				
				user.send({foundInFile: foundInFile});
				
			}
			
			if(replaceWith) {
				
				console.log("Replacing in file: " + filePath);
				
				fileContent = fileContent.replace(myRe, replaceWith);
				
				API.saveToDisk(user, {path: filePath, text: fileContent}, function readFile(err, json) {
					if(err) return abortError(err);
						
					filesBeingSearched--;
					totalFilesSearched++;
					doneMaybe();
					
				});
				
			}
			else {
				filesBeingSearched--;
				totalFilesSearched++;
				doneMaybe();
			}
			
			
		});
	}
	
	function doneFinish(msg) {
		if(done) throw new Error("Already done!");
		
		done = true;
		
		if(msg == undefined) {
			var totalTime = Math.round(((new Date()) - searchBegin) / 10) / 100;
			msg = "Found " + totalMatches + " match(es) in " + totalFiles + "/" + totalFilesFound + " file(s) searched in " + totalTime + "s.\n";
			}
		
		findReplaceInFilesCallback(null, {msg: msg, matches: matches});
		findReplaceInFilesCallback = null;
		
	}
	
	function aborted() {
		console.log("Aborting file search/replace");
		
		var msg = "Search was canceled! "
		var totalTime = Math.round(((new Date()) - searchBegin) / 10) / 100;
		msg = "Found " + totalMatches + " match(es) in " + totalFiles + "/" + totalFilesFound + " file(s) searched in " + totalTime + "s.\n";
		
		doneFinish(msg);
	}
	
	function abortError(err) {
		if(findReplaceInFilesCallback) {
findReplaceInFilesCallback(err);
			findReplaceInFilesCallback = null;
		}
		FIND_IN_FILES_ABORTED = true;
	}
	
	function sendProgress() {
		var now = new Date();
		if(now - lastProgress > progressInterval) {
			user.send({
				findInFilesStatus: {
					totalFoldersToSearch: totalFoldersToSearch,
					totalFoldersSearched: totalFoldersSearched,
					foldersBeingSearched: foldersToRead,
					fileQueue: fileQueue.length,
					totalFiles: totalFiles,
					totalFilesSearched: totalFilesSearched,
					filesBeingSearched: filesBeingSearched,
					totalMatches: totalMatches,
					maxTotalMatches: maxTotalMatches,
					searchString: searchString,
					folder: searchPath
				}
			});
			lastProgress = now;
		}
	}
	
}

API.findFiles = function findFiles(user, json, findFilesCallback) {
	/*
		Finds all files recursively in a folder.
		Keep searching, in parent folder, until maxResults have been found!
		Notify the client when switching folder
		*/
	
	if(FIND_FILES_ABORTED && FIND_FILES_IN_FLIGHT > 0) return finish(null, {buzy: true});
	
	FIND_FILES_ABORTED = false;
	
	var startFolder = json.folder;
	var findFile = json.name;
	var useRegexp = json.useRegexp || false;
	var ignore = json.ignore || [];
	var allowGlobbing = json.allowGlob; // If set to true it will automatically search parent folders
	
		if(startFolder == undefined) return finish(new Error("startFolder=" + startFolder));
		if(findFile == undefined) return finish(new Error("findFile=" + findFile));
		
		if(!useRegexp) findFile = UTIL.escapeRegExp(findFile);
		
		var reName = new RegExp(findFile, "ig");
		
		var maxResults = json.maxResults || 20;
	var maxConcurrency = 4; // 200
		var filesFound = 0;
		var foldersToSearch = [];
		var searchQueue = [];
		var lastProgress = new Date();
		var progressInterval = 350; // Prevent spamming the client when searching thousands of folders
		startFolder = UTIL.trailingSlash(startFolder);
		
		var folders = UTIL.getFolders(startFolder, true);
		var totalFoldersToSearch = 0;
		var totalFoldersSearched = 0;
		var callbackCalled = false;
	var foldersIgnored = 0;
	var filesIgnored = 0;
		var currentFolder = folders.pop();
		
		searchFolder(currentFolder);
		
		function searchFolder(folder) {
			if(folder == undefined) throw new Error("folder=" + folder);
			//console.log("FIND_FILES_IN_FLIGHT=" + FIND_FILES_IN_FLIGHT + " searchQueue.length=" + searchQueue.length + " folder=" + folder);
			if(FIND_FILES_ABORTED) return finish(null);
			if(foldersToSearch.indexOf(folder) != -1) return; // Folder already searched
			totalFoldersToSearch++;
			if(FIND_FILES_IN_FLIGHT >= maxConcurrency) {
				if(searchQueue.indexOf(folder) == -1) searchQueue.push(folder);
				sendProgress();
				return;
			}
			foldersToSearch.push(folder);
			FIND_FILES_IN_FLIGHT++;
			sendProgress();
		API.listFiles(user, {pathToFolder: folder}, function listFilesCallback(err, fileList) {
				FIND_FILES_IN_FLIGHT--;
				totalFoldersSearched++;
				
				if(FIND_FILES_ABORTED) return finish(null);
				
				if(err) {
					console.warn(err.message);
					return;
				}
				
				for (var i=0, path, matchArr; i<fileList.length; i++) {
					path = user.toVirtualPath(fileList[i].path);
					
				if(ignore.indexOf(path) != -1) {
					if(fileList[i].type=="d") foldersIgnored++;
					else filesIgnored++;
					continue;
				}
				
				if(fileList[i].type=="d") {
					// Do not search in dot files or temp/tmp folders
					if(fileList[i].name != "temp" && fileList[i].name != "tmp" && fileList[i].name.substr(0,1) != ".") {
						searchFolder(path);
						}
					}
					else {
						matchArr = path.match(reName);
						if(matchArr) {
							user.send({
								fileFound: {
									path: path, 
									match: matchArr,
									totalFoldersToSearch: totalFoldersToSearch,
									totalFoldersSearched: totalFoldersSearched,
									foldersBeingSearched: FIND_FILES_IN_FLIGHT,
									found: filesFound,
									maxResults: maxResults,
								}
							});
							filesFound++;
							if(filesFound >= maxResults) break;
						}
					}
				}
				
				sendProgress();
				
				/*
					console.log("filesFound=" + filesFound + " maxResults=" + maxResults + " FIND_FILES_IN_FLIGHT=" + FIND_FILES_IN_FLIGHT + 
					" searchQueue.length=" + searchQueue.length + " folders.length=" + folders.length + " folders=" + JSON.stringify(folders));
				*/
				
				if(filesFound >= maxResults) {
					finish(null);
				}
			else if(FIND_FILES_IN_FLIGHT < maxConcurrency && searchQueue.length > 0) {
				for (var i=FIND_FILES_IN_FLIGHT; i<maxConcurrency && searchQueue.length > 0; i++) searchFolder(searchQueue.pop());
			}
			else if(allowGlobbing && FIND_FILES_IN_FLIGHT == 0 && searchQueue.length == 0 && folders.length > 0) {
					currentFolder = folders.pop();
					user.send({pathGlob: currentFolder});
					searchFolder(currentFolder);
				}
				else if(FIND_FILES_IN_FLIGHT == 0 && searchQueue.length == 0) {
					finish(null);
				}
				else if(FIND_FILES_IN_FLIGHT == 0) throw new Error("Unexpected: FIND_FILES_IN_FLIGHT=" + FIND_FILES_IN_FLIGHT + " folders=" + JSON.stringify(folders) +
				" filesFound=" + filesFound + " maxResults=" + maxResults);
				
			});
		}
		
		function sendProgress() {
			var now = new Date();
			if(now - lastProgress > progressInterval) {
				user.send({
					findFilesStatus: {
						totalFoldersToSearch: totalFoldersToSearch,
						totalFoldersSearched: totalFoldersSearched,
						foldersBeingSearched: FIND_FILES_IN_FLIGHT,
						found: filesFound,
						maxResults: maxResults,
						name: findFile,
						folder: currentFolder
					}
				});
				lastProgress = now;
			}
		}
		
		function finish(err, resp) {
		if( (FIND_FILES_IN_FLIGHT != 0 || searchQueue.length > 0) && (!FIND_FILES_ABORTED && filesFound < maxResults && folders.length > 0) ) {
			throw new Error("FIND_FILES_IN_FLIGHT=" + FIND_FILES_IN_FLIGHT + " filesFound=" + filesFound + 
			" maxResults=" + maxResults + " folders.length=" + folders.length + " searchQueue.length=" + searchQueue.length + 
			"FIND_FILES_ABORTED=" + FIND_FILES_ABORTED);
		}
		
		FIND_FILES_ABORTED = true;
			if(!callbackCalled) {
				callbackCalled = true;
				if(!resp) resp = {};
				
				if(!resp.buzy) resp.buzy = false;
				if(!resp.found) resp.found = filesFound;
				if(!resp.foldersBeingSearched) resp.foldersBeingSearched = FIND_FILES_IN_FLIGHT;
				if(!resp.maxResults) resp.maxResults = maxResults;
				if(!resp.totalFoldersToSearch) resp.totalFoldersToSearch = totalFoldersToSearch;
				if(!resp.totalFoldersSearched) resp.totalFoldersSearched = totalFoldersSearched;
				if(!resp.name) resp.name = findFile;
			if(!resp.foldersIgnored) resp.foldersIgnored = foldersIgnored;
			if(!resp.filesIgnored) resp.filesIgnored = filesIgnored;
			
				findFilesCallback(err, resp);
			}
		}
		
	}

API.abortFindFiles = function abortFindFiles(user, json, abortFindFilesCallback) {
	FIND_FILES_ABORTED = true;
abortFindFilesCallback(null, {foldersBeingSearched: FIND_FILES_IN_FLIGHT});
	}

API.abortFindInFiles = function abortFindInFiles(user, json, callback) {
	FIND_IN_FILES_ABORTED = true;
	callback(null);
}

API.startProcess = function startProcess(user, json, callback) {
	// Starts a long running process
	
	var pArgs = json.args;
	var pPath = json.path;
	var pid = -1;
	var username = user.name;
	
	console.log(   "Starting " + pPath + " with args=" + JSON.stringify(pArgs) + " (" + pArgs.join(" ") + ")"   );
	var module_child_process = require("child_process");
	var p = module_child_process.spawn(pPath, pArgs);
	
	p.on("close", function (code, signal) {
		console.log(username + " " + pPath + " close: code=" + code + " signal=" + signal);
		user.send({process: {pid: pid, close: true, code: code}});
	});
	
	p.on("disconnect", function () {
		console.log(username + " " + pPath + " disconnect: p.connected=" + p.connected);
	});
	
	p.on("error", function (err) {
		console.log(username + " " + pPath + " error: err.message=" + err.message);
		console.error(err);
		user.send({process: {pid: pid, error: err.message, errorCode: err.code}});
	});
	
	p.stdout.on("data", function(data) {
		console.log(username + " " + pPath + " stdout: " + data);
		user.send({process: {pid: pid, stdout: data.toString()}});
	});
	
	p.stderr.on("data", function (data) {
		console.log(username + " " + pPath + " stderr: " + data);
		user.send({process: {pid: pid, stderr: data.toString()}});
	});
	
	pid = p.pid;
	
	PROCESS[pid] = p;
	
	callback(null, {pid: pid});
}

API.killProcess = function killProcess(user, json, callback) {
	// Kills a process
	
	var pid = parseInt(json.pid);
	
	if(typeof pid != "number") return callback(new Error("pid=" + pid + " json.pid=" + json.pid + " is not a number!"));
	if(pid < 1) return callback(new Error("pid=" + pid + " is less then 1"));
	
	if(PROCESS.hasOwnProperty(pid)) {
		PROCESS[pid].kill();
		delete PROCESS[pid];
		return callback(null);
	}
	else {
		if(!module_ps) return callback(new Error("Module ps not loaded!"));
		
		console.log("pid=" + pid + " not in PROCESS " + JSON.stringify(Object.keys(PROCESS)));
		module_ps.kill( pid, function( err ) {
			return callback(err);
		});
	}
}

API.run = function run(user, json, callback) {
	// Runs a shell command
	
	// Use exec instead of execFile because it's too hard to know how the arguments should be passed.
	var exec = require('child_process').exec;
	
	var options = {
		encoding: 'utf8',
		maxBuffer: 200*1024,
		env: process.env,
		shell: EXEC_OPTIONS.shell,
cwd: user.workingDirectory
	};
	
	/*
		env: {
		HOME: "/",
		PATH:"/bin/:/usr/bin/",
		USER: user.name,
		LOGNAME: user.name,
		uid:
		gid:
		}
	*/
	
	if(json.cwd) {
options.cwd = json.cwd;
	}
	else {
		options.cwd = user.workingDirectory;
		// If no cwd is specified __dirname (where this script is located) will be used!
	}
	
	if(json.env) {
		for(var prop in json.env) {
			options.env[prop] = json.env[prop];
		}
	}
	
	var command = json.command;
	
	console.log("Running command=" + command + " ...");
	console.log("env=" + JSON.stringify(options.env, null, 2));
	exec(command, options, function (err, stdout, stderr) {
		
		console.log(command + " => err=" + (err ? err.message : null) + " stdout=" + stdout + " stderr=" + stderr);
		
		if(err) {
			console.log("err.code=" + err.code);
			return callback(err);
		}
		else return callback(null, {stdout: stdout, stderr: stderr});
		
	});
}


/*
	
	API.shell = function shellCommand(user, json, shellCommandCallback) {
	// Deprecated! Use virtual terminal instead!
	var exec = require('child_process').exec;
	
	var commandToRun = json.command;
	
	var execOptions = {
		encoding: 'utf8',
		timeout: 2000,
		maxBuffer: 200*1024,
		killSignal: 'SIGTERM',
		cwd: null,
		env: null,
shell: EXEC_OPTIONS.shell
	}
	
	exec(commandToRun, execOptions, function(err, stdout, stderr) {
		var output = stdout + stderr;
		
		if(typeof output == "string") output = output.replace(/\r/g, "");
		
		shellCommandCallback(err, {output: output});
		
	});
	}
*/

API.ping = function ping(user, json, callback) {
	callback(null, json);
}

API.cpu = function cpu(user, json, callback) {
	var os = require('os');
	callback(null, os.cpus());
}

API.memory = function memory(user, json, callback) {
	var os = require('os');
	
	callback(null, {total: os.totalmem(), free: os.freemem()});
}

API.platform = function platform(user, json, callback) {
	callback(null, process.platform);
}

API.crashme = function crash(user, json, callback) {
	// Only used to test what happens if the user worker crashes
	throw new Error("Crash boom bang!");
}

function runFtpQueue() {
	
	console.log(ftpQueue.length + " items left in the FTP queue");
	
	if(ftpQueue.length > 0) {
		console.log("Executing next item in the ftp queue ...");
		ftpQueue.shift()();
	}
	else ftpBusy = false;
	
}




module.exports = API;
