#!/bin/sh
':' //; exec "$(command -v nodejs || command -v node)" "$0" "$@"

/*
	Universal editor start script for running locally (desktop)
	
	Usage:
	node start.js [options]
	
	options:
	--loglevel={1-7}  Set the log level
	-local            Don't search for servers
	-debug            Same as --loglevel=7)
	-no-module-check  Don't check if "npm install" have been run
	
	--------------------------------------------------------------------------------
	
	First try to find a server on localhost
	Then listen to editor broadcasts on the LAN.
	But if no server is found, start one on localhost
	
	Then try to run the client in nw.js
	But if that fails launch the client in a browser: Chrome, Firefox, IE or Safari
	
	Debug:
	node start.js -debug > debug.log 2>&1
	
	Optimization notes:
	start client: ca 450ms
	start server: ca 240ms
	
*/

"use strict";

var module_log = require("./shared/log.js")
var log = module_log.log;

var module_child_process = require('child_process');

// Log levels
var WARN = 4;
var NOTICE = 5;
var INFO = 6;
var DEBUG = 7;

console.log("./start.js args=" + JSON.stringify(process.argv));

var getArg = require("./shared/getArg.js");

var LOG_LEVEL = getArg(["loglevel", "loglevel"]) || INFO;
if(!!getArg(["debug", "debug"])) LOG_LEVEL = DEBUG;

module_log.setLogLevel(LOG_LEVEL);
module_log.overrideConsole();

var LOCAL_SERVER_IP = "127.0.0.1";
var LOCAL_SERVER_PORT = "8099";

var localOnly = !!getArg(["local", "local"]);



var serverFound = false;
var clientStarting = false;
var serversChecked = 0;
var serversToCheck = 0;

var HTTP_REQUESTS = [];
//return startClient("127.0.0.1", "8099");

var module_path = require("path");

var serverProcess; // So it can be killed when client exit
var clientProcess; // So we can kill it at will

var clientConnected = false; // Will be set to true when the server detects a HTTP request

var no_module_check = !!getArg(["no-module-check", "no-module-check"]);
console.log("no_module_check=" + no_module_check);

if(no_module_check) {
	console.log("Skiping module check");
	modulesChecked();
}
else checkModules();

function modulesChecked() {
	if(localOnly) {
		startClient();
		startNewServer();
	}
else {
		// We would like to start the client before starting the server because it will take longer to start
		// But the client needs to know which server and port to connect to, we don't know that until we have found a server.
		// Use the -local flag for faster startup!
		
	// Check the network for servers (but first check localhost)
	// And if no server is found withing 1? seconds we'll start our own server
	setTimeout(startNewServer, 1000); // 3000ms
	
	log("Check if server is running on localhost ...", INFO);
	checkServer(LOCAL_SERVER_IP, serverChecked);
	
	var adresses = getIpv4Ips();
	for(var i=0; i<adresses.length; i++) checkServer(adresses[i], serverChecked);
}
}

function checkModules() {
	// Checking if the modules/dependencies exist is neccesary because npm *sometimes* doesn't install them ...
	// The following require takes ca 70ms to run
	
	console.log("Check if node modules are installed ...");
	console.time("checkModules");
	var error = undefined;
	var moduleName = "sockjs";
	try {
		require(moduleName);
	}
	catch(err) {
		error = err;
	}
	
	if(error) {
		console.timeEnd("checkModules");
		console.log("Unable to require " + moduleName + " module: " + error.message);
		
		// Use spawn instead of exec so we can see the progress bar
		var spawn = module_child_process.spawn;
		var arg = ["install"];
		var options = {
			cwd: __dirname,
			stdio: ['inherit', 'inherit', 'inherit'],
			shell: true
		};
		console.log("Running npm " + arg[0] + " ...");
		var npm = spawn("npm", arg , options);
		
		npm.on('error', function npmError(err) {
			log("npm " + arg[0] + " Error: " + err.message);
		});
		
		npm.on('close', function npmClose(exitCode) {
			console.log("npm " + arg[0] + " exitCode=" + exitCode + " ");
			
			log("Finished installing dependencies.");
			
			// Exit here so that the use can see possible npm errors ??
			modulesChecked();
			
			//log("Try running the program again!", NOTICE); // it might work even if there where npm errors!
			//process.exit();
		});
		
	}
	else {
		console.timeEnd("checkModules");
		console.log("Dependenices/modules seem to be installed.");
		modulesChecked();
	}
}

function serverChecked(online, ip, port) {
	//log("server ip:" + ip + " is", online ? "ON" : "offline");
	
	serversChecked++;
	
	log("serversChecked=" + serversChecked + " serversToCheck=" + serversToCheck + " serverFound=" + serverFound, DEBUG);
	
	
	if(online) {
		log("Found server running on ip=" + ip + " port=" + port, INFO);
		
		serverFound = true;
		
		startClient(ip, port);
		abortHttpRequests();
		
		
	}
	else {
		if(serversChecked == serversToCheck && !serverFound) broadcast();
	}
}



function abortHttpRequests() {
	for(var i=0; i<HTTP_REQUESTS.length; i++) HTTP_REQUESTS[i].abort();
}


function startNewServer() {
	if(serverFound) return;
	
	console.time("startNewServer");
	
	abortHttpRequests();
	
	serverFound = true;
	
	log("Starting new server ...");
	
	var serverIp = LOCAL_SERVER_IP;
	var serverPort = LOCAL_SERVER_PORT;
	
	var scriptPath = module_path.resolve(__dirname, "server/server.js");
	var serverArg = [scriptPath, "--loglevel=" + LOG_LEVEL, "--username=admin", "--password=admin", "--ip=" + serverIp, "--port=" + serverPort];
	
	var serverOptions = {
		stdio: "inherit"
	}
	
	attemptLaunch("node", serverArg, serverOptions, function(err, cp) {
		console.timeEnd("startNewServer");
		if(err) log("Unable to start server!");
		else {
			
			log("Server started!");
			
			if(!cp) throw new Error("Got no server child process!");
			serverProcess = cp;
			
			if(!clientStarting) startClient();
			
			serverProcess.stdout.on("data", serverLog);
			serverProcess.stderr.on("data", serverLog);
		}
		
		var lastServerLogMsg = new Date();
		
		function serverLog(data) {
			lastServerLogMsg = new Date();
			if(typeof data == "object") data = data.toString();
			if(data.match(/Closed client connection/)) {
				console.log("Detected: Closed client connection.");
				
				setTimeout(function() {
					if( ((new Date()) - lastServerLogMsg) > 1000 ) { 
						console.log("Killing the server because there have been no server log messages for a second ...");
				serverProcess.kill();
				console.log("Exiting because the client disconnected from the server and the server was killed.");
				process.exit(0);
					}
					else console.log("The Close client connection was probably a reload");
				}, 2000);
				
			}
			else if(!clientConnected && data.match(/HTTP-req/)) {
clientConnected = true;
				console.timeEnd("startClient");
			}
		}
	});
}

function broadcast() {
	// Listen to and send broadcast messages asking for webide server
	// http://stackoverflow.com/questions/6177423/send-broadcast-datagram
	
	var broadcastPort = 6024;
	var myIps = getIpv4Ips();
	var broadcastAddresses = myIps.map(broadcastAddress);
	
	console.log("broadcastAddresses: ", broadcastAddresses);
	
	var dgram = require('dgram');
	
	// Server
	var broadcastServer = dgram.createSocket("udp4");
	broadcastServer.bind(function() {
		broadcastServer.setBroadcast(true);
		// We must send at least one broadcast message to be able to receive messages!
		for(var i=0; i<broadcastAddresses.length; i++) ask(broadcastAddresses[i]);
	});
	
	// Client
	var broadcastClient = dgram.createSocket('udp4');
	
	broadcastClient.on('listening', function () {
		var address = broadcastClient.address();
		console.log('UDP Client listening on ' + address.address + ":" + address.port);
		broadcastClient.setBroadcast(true);
	});
	
	broadcastClient.on('message', function (message, rinfo) {
		console.log('Message from: ' + rinfo.address + ':' + rinfo.port +' - ' + message);
		
		
		// webide server url: http://127.0.0.1/
		// webide server url: http://127.0.0.1:8099/
		
		message = message.toString("utf8");
		
		var matchUrl = message.match(/webide server url: (https?):\/\/(\d{1,3}.\d{1,3}.\d{1,3}.\d{1,3}):?(\d*)?/);
		
		if(matchUrl) {
			var proto = matchUrl[1];
			var ip = matchUrl[2];
			var port = matchUrl[3];
			
			serverFound = true;
			startClient(ip, port, proto);
			
			//clearInterval(askForServerInterval);
			
			broadcastClient.close();
			broadcastServer.close();
			
		}
		
	});
	
	broadcastClient.bind(broadcastPort);
	
	function ask(broadcastAddress) {
		var lookForServerMessage = "Where can I find a webide server?"
		var message = new Buffer(lookForServerMessage);
		broadcastClient.send(message, 0, message.length, broadcastPort, broadcastAddress, function() {
			console.log("Sent '" + message + "'");
		});
	}
	
	function broadcastAddress(ip) {
		// Asume 255.255.255.0 netmask
		var arr = ip.split(".");
		arr[3] = "255";
		return arr.join(".");
	}
}

function getIpv4Ips() {
	var os = require('os');
	
	var interfaces = os.networkInterfaces();
	var addresses = [];
	for (var k in interfaces) {
		for (var k2 in interfaces[k]) {
			var address = interfaces[k][k2];
			if (address.family === 'IPv4' && !address.internal) {
				addresses.push(address.address);
			}
		}
	}
	
	return addresses;
}

function isPrivatev4IP(ip) {
	var parts = ip.split('.');
	return parts[0] === '10' || parts[0] === '127' ||
	(parts[0] === '172' && (parseInt(parts[1], 10) >= 16 && parseInt(parts[1], 10) <= 31)) || 
	(parts[0] === '192' && parts[1] === '168');
}

function checkServer(ip, callback) {
	
	serversToCheck++;
	
	if(typeof callback != "function") throw new Error("callback=" + callback + " need to be a callback function!");
	
	if(serverFound) {
		//log("A server has already been found. Aborting checkServer", DEBUG);
		return;
	}
	
	log("Checking for a webide server on ip=" + ip + " ...", DEBUG);
	
	var http = require("http");
	
	var portFound = false;
	var portsChecked = 0;
	
	var portsToCheck = [80, 8080, 8099];
	
	for(var i=0; i<portsToCheck.length; i++) checkPort(portsToCheck[i], portChecked);
	
	function portChecked(itsTheServer, port) {
		portsChecked++;
		
		if(itsTheServer) {
			portFound = true;
			callback(true, ip, port);
		}
		else if(portsChecked == portsToCheck.length && !portFound) callback(false, ip);
	}
	
	function checkPort(port, checkPortCallback) {
		
		if(serverFound) {
			//log("A server has already been found. Aborting checkServer checkPort", DEBUG);
			return;
		}
		
		log("Checking port=" + port + " on ip=" + ip, DEBUG);
		
		
		var options = {
			host: ip,
			port: port,
			path: '/webide',
			method: 'GET'
		};
		
		var req = http.request(options, function(res) {
			
			if(serverFound) return;
			
			log("Answer on port=" + port + " on ip=" + ip, INFO);
			log('STATUS: ' + res.statusCode, DEBUG);
			log('HEADERS: ' + JSON.stringify(res.headers), DEBUG);
			res.setEncoding('utf8');
			var body = "";
			res.on('data', function (chunk) {
				log('BODY: "' + chunk + '"', DEBUG);
				body += chunk;
			});
			res.on("end", function(chunk) {
				log('END: body="' + body + '"', DEBUG);
				if(body == "Welcome to SockJS!\n") checkPortCallback(true, port);
				else checkPortCallback(false, port);
			});
		});
		
		HTTP_REQUESTS.push(req);
		
		req.on('error', function(e) {
			if(!serverFound) log('problem with request: ' + e.message, DEBUG);
			checkPortCallback(false, port);
		});
		
		req.end();
	}
}

function timeStamp() {
	return (new Date()).getTime()/1000|0; // Unix timestamp
}

function startClient(ip, port, proto) {
	
	if(clientStarting) {
		log("Client already about to start!");
		return;
	}
	clientStarting = true;
	log("Starting client ...");
	console.time("startClient");
		
		
	if(port == undefined) port = LOCAL_SERVER_PORT;
	if(ip === undefined) ip = LOCAL_SERVER_IP;
	
	var portPart = "";
	
	if(port != undefined && port != "80") portPart = ":" + port;
	
	if(proto == undefined) proto = "http";
	var url = "http://" + ip + portPart + "/";
	
	var nwRuntime = "";
	var platform = process.platform;
	if(platform == "darwin") nwRuntime = "./runtime/nwjs-v0.12.3-osx-x64/nwjs.app/Contents/MacOS/nwjs";
	else if(platform == "win32")  nwRuntime = "./runtime/nwjs-v0.12.3-win-x64/nw.exe";
	else if(platform == "linux")  nwRuntime = "./runtime/nwjs-v0.12.3-linux-x64/nw";
	else log("platform=" + platform + " not yet supported by nw.js", INFO);
	
	
	var tryPrograms = [];
	
	if(platform == "win32") {
		// Only try IE on Windows
		tryPrograms.push(["cmd", ["/K", "start", '""', "iexplore", "-k", url]]);
	}
	else if(platform == "darwin") {
		tryPrograms.push(['"/Applications/Google\ Chrome.app/Contents/MacOS/Google\ Chrome"', ['--app="' + url + '"']]);
		
		// Unfortunately Safari doesn't support chromless
		// We might be able to remove the chrome after it started though, by using osascript
		
		//tryPrograms.push(["/Applications/Safari.app/Contents/MacOS/Safari & sleep 1 && osascript -e 'tell application \"Safari\" to open location \"http://www.google.com\"'"]);
		
		tryPrograms.push(["open", ["-W", "-a", "safari", url]]);
	}
	else {
		// Always try nw.js first!
		//tryPrograms.push(["nw", ["."]]); // Any version of nw.js
		//tryPrograms.push([nwRuntime, ["."]]); // The included nw.js runtime
		
		// We prefer the chromium/chrome browser!
		tryPrograms.push(["chrome", ["--app=" + url, "--disable-gpu-vsync "]]);
		tryPrograms.push(["chromium-browser", ["--app=" + url, "--disable-gpu-vsync "]]);
		
		// It seems Firefox doesn't want to open URL's in chromeless mode (-chrome), only files
		// We want to open files via http/https though! Using file:// protocol will cause issues.
		tryPrograms.push(["firefox", ["-new-tab", url]]); // We can open a url in a new tab though
		//tryPrograms.push(["firefox", ["-chrome", "client/index.htm"]]);
	}
	
	
	var programIndex = 0;
	var startTime = timeStamp();
	var maxTime = 3; // Seconds
	var programStarted = false;
	
	tryProgram(tryPrograms[programIndex]);
	
	
	
	function tryProgram(arr) {
		var program = arr[0];
		var args = arr[1] || [];
		
		attemptLaunch(program, args, function triedProgram(err, cp) {
			if(err) {
				
				log("Failed to start program=" + program);
				
				var time = timeStamp();
				
				if(time - startTime > maxTime && programStarted) {
					log((time - startTime) + " seconds since start. Assuming exit");
					return process.exit();
				}
				
				programIndex++;
				if(programIndex >= tryPrograms.length) {
//throw new Error("Unable to start browser engine!");
					log("Manually open a browser, and visit the following address/URL:\n" + url);
				}
				else tryProgram(tryPrograms[programIndex]);
			}
			else {
					// Depending on the program we wont get a callback until the browser/runtime has already exited!
				log("Successfully started program=" + program);
				log("If the program however failed to start,");
				log("open your favorite browser and navigate to the URL below:");
				log(proto + "://" + ip + ":" + port);
				
				programStarted = true;
				
				if(cp) { // Don't check if the cp is connected (it's not)
					clientProcess = cp;
					cp.on("close", function killServer(code) {
						if(serverProcess) {
						console.log("Killing server process because " + program + " closed! (code=" + code + ")");
						serverProcess.kill();
						serverProcess.unref();
						}
						else console.log("serverProcess not running !? (Got cp)");
						
						console.log("Exiting because client process closed!");
						process.exit(0); // Exit start script when client closes
					});
					
				}
				//else if(!serverProcess) console.warn("We do not yet have the server process!"); 
				else {
					console.log("Did not recive cp after launching " + program);
					// No child-process in callback means we got the callback on the close event
					// Kill the server right away
					if(serverProcess) {
						console.log("Killing server process because cp=" + !!cp + ""); // and cp.connected=" + (cp && cp.connected) + "
						serverProcess.kill();
						serverProcess.unref();
					}
					else console.log("serverProcess not running !? (Did not get cp)");
					
					console.log("Exiting because cp=" + !!cp);
					process.exit(0); // Exit start script when client closes
				}
				
			}
		});
	}
}

function attemptLaunch(program, args, options, uid, gid, callbackFunction) {
	
	if(program == undefined) throw new Error("No program specified");
	if(!Array.isArray(args)) throw new Error("No program argiments specified");
	if(typeof options == "function") {
		callbackFunction = options;
		options = undefined;
	}
	if(typeof options == "function") {
		callbackFunction = options;
		options = undefined;
	}
	if(typeof uid == "function") {
		callbackFunction = uid;
		uid = undefined;
	}
	
	if(options && typeof options != "object") throw new Error("options need to be an key:value object");
	if(uid != undefined && typeof uid != "number") throw new Error("uid needs to be a numeric value");
	if(gid != undefined && typeof gid != "number") throw new Error("gid needs to be a numeric value");
	if(typeof callbackFunction != "function") throw new Error("No callback function!");
	
	var callback = function(err, cp) {
		if(callbackFunction) callbackFunction(err, cp);
		callbackFunction = null; // Only callback once!
	}
	
	log("Attemting to start program=" + program + " args=" + JSON.stringify(args) + " uid=" + uid + " gid=" + gid, DEBUG);
	
	// You can have different group and user. Default is the user/group running the node process
	var options = {};
	
	var gotStdoutData = false;
	
	if(uid != undefined) options.uid = parseInt(uid);
	if(gid != undefined) options.gid = parseInt(gid);
	
	if(process.platform=="darwin") options.shell = true; // Use sh -c on Mac for Chrome to work
	
	try {
		var cp = module_child_process.spawn(program, args, options);
	}
	catch(err) {
		if(err.code == "EPERM") {
			if(uid != undefined) log("Unable to spawn program=" + program + " with uid=" + uid + " and gid=" + gid + ".\nTry running the script with a privileged (sudo) user.", NOTICE);
		}
		var msg = "Unable to spawn program! (" + err.message + ")";
		log(msg, DEBUG)
		return callback(new Error(msg));
	}
	
	cp.ref(); // Do not uncouple!
	
	if(cp.connected && callbackFunction) {
		log("Assuming program=" + program + " was successful because it's connected!", DEBUG);
		return callback(null, cp);
	}
	
	cp.on("close", function programClose(code, signal) {
		var msg = program + " close: code=" + code + " signal=" + signal;
		log(msg, DEBUG);
		
		code = parseInt(code);
		if(code === 0 && callbackFunction) {
			log("Assuming program=" + program + " was successful because close code=" + code, DEBUG);
			callback(null); // Don't return child-process after it has closed
		}
		else callback(new Error(msg));
		
	});
	
	cp.on("disconnect", function programDisconnect() {
		var msg = program + " disconnect: cp.connected=" + cp.connected;
		log(msg, DEBUG)
		callback(new Error(msg));
	});
	
	cp.on("error", function programClose(err) {
		var msg = program + " error: err.message=" + err.message
		log(msg, DEBUG);
		callback(new Error(msg));
	});
	
	cp.on("exit", function programExit(code, signal) {
		var msg = program + " exit: code=" + code + " signal=" + signal;
		log(msg, DEBUG);
	});
	
	cp.stdout.on("data", function programStdout(data) {
		var msg = program + " stdout data: " + data;
		log(msg, DEBUG);
		
		if(!gotStdoutData && callbackFunction) {
			gotStdoutData = true;
			log("Assuming program=" + program + " was successful because something was returned from stdout!", DEBUG);
			callback(null, cp);
		}
		
	});
	
	cp.stderr.on("data", function programStderr(data) {
		var msg = program + " stderr data: " + data;
		log(msg, DEBUG);
		
		// Node -v 8 seems to get all data on stderr instead of the correct stdout ... 
		if( !gotStdoutData && callbackFunction && (program == "node" || program == "nodejs") ) {
			gotStdoutData = true;
			log("Assuming program=" + program + " was successful because something was returned from stderr!", DEBUG);
			callback(null, cp);
		}
		
	});
	
	/*
		var waitTime = 250;
		setTimeout(function started() {
		log("Assuming program=" + program + " successful because nothing happened within " + waitTime + "ms!");
		callback(null);
		}, waitTime);
	*/
	
}
